/*----------------------------------------------------------------------
MODEL prescaler.sv

= Purpose =

A prescaler with a parameterized division factor.

= Description =

This prescaler is triggered by both the rising and falling edges of the input clock in order to keep the output duty cycle close to 50% even for odd division factors. For the dividing factor, 2, 4, and 8 are available.

A fixed-timestep model.

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module prescaler #(
    `parameter_real(delay_cq_0, 225e-12), // clock-to-output delay 0
    `parameter_real(delay_cq_1, 332e-12), // clock-to-output delay 1
    `parameter_real(delay_cq_2, 405e-12), // clock-to-output delay 2
    `parameter_real(cap_in, 1.8e-15),     // equivalent input capacitance
    `parameter_real(res_out, 0.52e4)      // equivalent output resistance
)(
    input vdd,
    input vss,
    input reset_n,          // reset signal (active low)
    input [1:0] div_sel,    // dividing factor selection code (2,4,8)
    `input_xbit clk_inp,    // input clock (positive)
    `input_xbit clk_inn,    // input clock (negative)
    `output_xbit clk_outp,  // output clock (positive)
    `output_xbit clk_outn   // output clock (negative)
);

    reg [1:0] div_sel_retimed;
    xbit [1:0] div_sel_xbit;
    xbit rst;
    xbit clk_div2, clk_div2_n, clk_div4, clk_div4_n, clk_div8, clk_div8_n;
    xbit clk_div2_out, clk_div4_out, clk_div8_out;
    xbit clk_div2_out_n, clk_div4_out_n, clk_div8_out_n;

    always @(posedge `value(clk_outp) or negedge reset_n) begin
        if(!reset_n) begin
            div_sel_retimed <= 2'b00;
        end
        else begin
            div_sel_retimed <= div_sel;
        end
    end

    bit_to_xbit #(.width(2)) conn0(.in(div_sel_retimed), .out(div_sel_xbit));
    bit_to_xbit conn1(.in(~reset_n), .out(rst));

    dff_rst_async_xbit
        dff0(.d(clk_div2), .q(clk_div2_n), .clk(clk_inp), .rst(rst));
    inv_xbit inv0(.in(clk_div2_n), .out(clk_div2));

    dff_rst_async_xbit
        dff1(.d(clk_div4), .q(clk_div4_n), .clk(clk_div2_n), .rst(rst));
    inv_xbit inv1(.in(clk_div4_n), .out(clk_div4));

    dff_rst_async_xbit
        dff2(.d(clk_div8), .q(clk_div8_n), .clk(clk_div4_n), .rst(rst));
    inv_xbit inv2(.in(clk_div8_n), .out(clk_div8));

    delay_xbit #(.delay(delay_cq_0))
        delay_0(.in(clk_div2), .out(clk_div2_out));
    delay_xbit #(.delay(delay_cq_0))
        delay_0b(.in(clk_div2_n), .out(clk_div2_out_n));
    delay_xbit #(.delay(delay_cq_1))
        delay_1(.in(clk_div4), .out(clk_div4_out));
    delay_xbit #(.delay(delay_cq_1))
        delay_1b(.in(clk_div4_n), .out(clk_div4_out_n));
    delay_xbit #(.delay(delay_cq_2))
        delay_2(.in(clk_div8), .out(clk_div8_out));
    delay_xbit #(.delay(delay_cq_2))
        delay_2b(.in(clk_div8_n), .out(clk_div8_out_n));

    mux_xbit #(.num_in(4), .width_sel(2))
        mux0(.sel(div_sel_xbit), .in({clk_div8_out,clk_div8_out,clk_div4_out,clk_div2_out}), .out(clk_outp));
    mux_xbit #(.num_in(4), .width_sel(2))
        mux1(.sel(div_sel_xbit), .in({clk_div8_out_n,clk_div8_out_n,clk_div4_out_n,clk_div2_out_n}), .out(clk_outn));

endmodule
