/*----------------------------------------------------------------------
MODEL bbpfd.sv

= Purpose =

A bang-bang phase-frequency detector model to compare the phase or frequency differences between the two input clocks, providing the polarity information only.

= Description =

The bang-bang phase-frequency detector measures the phase or frequency differences between the two input clocks and expresses its polarity in binary outputs (up and down).

= Revisions =

$Author$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module bbpfd #(
    `parameter_real(delay_rst, 131e-12),    // reset propagation delay
    `parameter_real(delay_cq, 311e-12),     // clock-to-output delay
    `parameter_real(delay_cp, 199e-12),     // clock-to-proportional path output delay
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 2.1e4),        // equivalent output resistance
    `parameter_real(res_out_prop, 0.26e4)   // equivalent output resistance (proportional term)
)(
    input vdd,
    input vss,
    `input_xbit clk_ref,    // reference clock input
    `input_xbit clk_fb,     // feedback clock input
    output up,              // output (up)
    output dn,              // output (down)
    `output_xbit prop_up,   // PFD output pulse (up)
    `output_xbit prop_dn    // PFD output pusle (dn)
);

    // parameters
    parameter real prop_up_delay = delay_cp;
    parameter real prop_dn_delay = delay_cp;

    // variables
    xbit up_l, dn_l, up_m, dn_m, upb_m, dnb_m, up_n, dn_n;
    xbit up_pre_xbit, dn_pre_xbit, up_xbit, dn_xbit;

    // phase-frequency detector
    bbpfd_pfd #(.delay_rst(delay_rst))
        pfd(.clk_ref(clk_ref), .clk_fb(clk_fb),
            .up(up_l), .dn(dn_l));

    // connectors
    xbit_to_bit xbit2bit0(.in(up_xbit), .out(up));
    xbit_to_bit xbit2bit1(.in(dn_xbit), .out(dn));

    // S'R'-NAND-latch : preceding edge detection
    nand_xbit latch_00(.in({up_l, dn_m}), .out(up_m));
    nand_xbit latch_01(.in({dn_l, up_m}), .out(dn_m));

    // gated-S'R'-NAND-latch : edge discrimination & retiming
    inv_xbit #(.delay(15e-12)) inv0(.in(up_m), .out(upb_m));
    inv_xbit #(.delay(15e-12)) inv1(.in(dn_m), .out(dnb_m));
    nand_xbit clk_gate0(.in({upb_m,clk_fb}), .out(up_n));
    nand_xbit clk_gate1(.in({dnb_m,clk_fb}), .out(dn_n));

    nand_xbit latch_10(.in({up_n,dn_pre_xbit}), .out(up_pre_xbit));
    nand_xbit latch_11(.in({dn_n,up_pre_xbit}), .out(dn_pre_xbit));
    
    delay_xbit #(.delay(delay_cq)) up_delay0(.in(up_pre_xbit), .out(up_xbit));
    delay_xbit #(.delay(delay_cq)) dn_delay0(.in(dn_pre_xbit), .out(dn_xbit));

    // PFD output pulse
    delay_xbit #(.delay(prop_up_delay)) up_delay(.in(up_l), .out(prop_up));
    delay_xbit #(.delay(prop_dn_delay)) dn_delay(.in(dn_l), .out(prop_dn));

endmodule

module bbpfd_pfd #(
    `parameter_real(delay_rst, 0.0)     // reset propagation delay
)(
    `output_xbit up,	                // up signal
    `output_xbit dn,	                // down signal
    `input_xbit clk_ref,                // reference clock
    `input_xbit clk_fb                  // feedback clock
);

    // variables
    xbit rst;

    // two dffs with asynchronous reset
    dff_rst_async_xbit #(.init_value(0)) dff_up(.q(up), .d(`one_xbit), .clk(clk_ref), .rst(rst));
    dff_rst_async_xbit #(.init_value(0)) dff_dn(.q(dn), .d(`one_xbit), .clk(clk_fb), .rst(rst));

    // reset path
    and2_xbit #(.delay(delay_rst)) and_xbit(.out(rst),.in_a(up),.in_b(dn));

endmodule
