/*----------------------------------------------------------------------
MODEL dpll.sv

= Purpose =

dpll model

= Description =

GRO_TDC + D-LF + LCDCO + PRESCALER + DIV8_MPHASE + PH_ROTATOR

= Revisions =

$Authors$
$DateTime$
$Id$

----------------------------------------------------------------------*/

module dpll (
    `input_xbit clk_ref,    // Reference clock from pulse generator
    input reset_n,          // Global reset (active low)
    input vdda,      // Analog power
    input vddt,      // Digital power
    input vss,       // Ground

    input [1:0] tdc_f_range,        // GRO_TDC operating frequency range
    input [1:0] ph_rot_f_range,     // PH_ROTATOR operating frequency range
    input [2:0] pgain_ctrl,         // proportional gain cotrol (DCO)
    input [1:0] div_sel_prescaler,  // dividing factor selection code in PRESCALER
    input [6:0] phsel,              // phase selection code in PH_ROTATOR

    `output_xbit clk_fb,            // Feedback clock(to pad) of PLL
    `output_xbit clk_outp,          // Output clock(to pad) of PLL (positive)
    `output_xbit clk_outn           // Output clock(to pad) of PLL (negative)
);

    // variables
    wire [5:0] up, dn;
    wire clk_lf;
    reg clk_lf_dco;
    reg [9:0] dctrl;
    reg [2:0] dctrl_dsm;

    `xbit(clk_fb_n);
    `xbit(prop_up);
    `xbit(prop_dn);
    `xbit(clk_div_pre_p);
    `xbit(clk_div_pre_n);
    `xbit([7:0] clk_div8_mphase);

    // 1. GRO_TDC --------------------------------------------------------
    gro_tdc gro_tdc (
        .vdd(vdda), .vss(vss),
        .f_range(tdc_f_range),
        .up(up), .dn(dn), .prop_up(prop_up), .prop_dn(prop_dn),
        .clk_ref(clk_ref), .clk_fb(clk_fb), .reset_n(reset_n)
    );

    // 2. digital filter (D-LF) ------------------------------------------
    digital_lf digital_lf (
        .up(up), .dn(dn), .reset_n(reset_n), .clk(clk_lf),
        .out(dctrl), .dsm_out(dctrl_dsm), .clk_lf_dco(clk_lf_dco)
    );

    `ifndef SYN
        xbit_to_bit conn(.in(clk_fb), .out(clk_lf));
    `else
        assign clk_lf = clk_fb;
    `endif

    // 3. LCDCO ----------------------------------------------------------
    lcdco lcdco (
        .vdd(vdda), .vss(vss),
        .clk_lf(clk_lf_dco),
        .dco_pgain(pgain_ctrl), .prop_up(prop_up), .prop_dn(prop_dn),
        .in(dctrl), .in_dsm(dctrl_dsm),
        .reset_n(reset_n), .clk_outp(clk_outp), .clk_outn(clk_outn)
    );

    // 4. PRESCALER ------------------------------------------------------
    prescaler prescaler (
        .vdd(vdda), .vss(vss),
        .clk_inp(clk_outp), .clk_inn(clk_outn),
        .clk_outp(clk_div_pre_p), .clk_outn(clk_div_pre_n),
        .reset_n(reset_n), .div_sel(div_sel_prescaler)
    );

    // 5. DIV8_MPHASE ----------------------------------------------------
    div8_mphase div8_mphase (
        .vdd(vdda), .vss(vss),
        .clk_inp(clk_div_pre_p), .clk_inn(clk_div_pre_n),
        .clk_out(clk_div8_mphase),
        .reset_n(reset_n)
    );

    // 6. PH_ROTATOR -----------------------------------------------------
    ph_rotator ph_rotator (
        .vdd(vdda), .vss(vss),
        .reset_n(reset_n),
        .f_range(ph_rot_f_range),
        .clk_in(clk_div8_mphase), .phsel(phsel),
        .clk_out(clk_fb), .clk_outb(clk_fb_n)
    );

endmodule
