/*----------------------------------------------------------------------
TESTBENCH tb_phasestep.sv

= Purpose =

A testbench to observe the phase step response of a digital PLL.

= Description =

A combination of step_gen and phase_to_clk generates a clock signal which has phase step and feeds it to dpll. The recovered clock is probed to observe its response.

= Revisions =

$Author$
$DateTIme$
----------------------------------------------------------------------*/

`include "xmodel.h"

module tb_phasestep();	

    parameter real ref_freq = 125.0e6;  // reference frequency
    parameter real phase_step = 0.1;    // phase step amount
    parameter real delay = 20.0e-6;     // A point to make phase step

    // variables
    bit reset_n;        // reset (active low)

    reg [1:0] tdc_f_range;
    reg [1:0] ph_rot_f_range;
    reg [2:0] pgain_ctrl;
    reg [1:0] div_sel_prescaler;
    reg [6:0] phsel;

    xbit clk_ref, clk_fb;
    xbit clk_outp, clk_outn;

    reg vdda = 1;
    reg vddt = 1;
    reg vss = 0;

    // phase step generator
    `xreal(phase);      // phase with step change
    step_gen     #(.init_value(0.0), .change(phase_step), .delay(delay))
                step_gen(phase);

    // reference clock generator
    phase_to_clk #(.freq(ref_freq)) clk_ref_gen(.in(phase), .out(clk_ref));

    initial begin
        reset_n = 1'b0;
        tdc_f_range = 2'b10;
        ph_rot_f_range = 2'b01;
        pgain_ctrl = 3'b111;
        div_sel_prescaler = 2'b10;
        phsel = 7'b0;
        #(1e-9/`TIME_SCALE) reset_n = 1'b0;
        #(2e-9/`TIME_SCALE) reset_n = 1'b1;
    end

    // DUT (device-under-test) digital PLL
    dpll
            DUT(.clk_ref(clk_ref), .reset_n(reset_n),
                .vdda(vdda), .vddt(vddt), .vss(vss),
                .tdc_f_range(tdc_f_range),
                .ph_rot_f_range(ph_rot_f_range),
                .pgain_ctrl(pgain_ctrl),
                .div_sel_prescaler(div_sel_prescaler),
                .phsel(phsel),
                .clk_fb(clk_fb),
                .clk_outp(clk_outp), .clk_outn(clk_outn));

    // probing
    probe_xbit  probe_ref(clk_ref);
    probe_xbit  probe_fb(clk_fb);
    probe_xbit  probe_outp(clk_outp);
    probe_xbit  probe_outn(clk_outn);
    probe_phase #(.freq(ref_freq), .phase_wrap(0)) probe_clk_ref(clk_ref);
    probe_phase #(.freq(ref_freq), .phase_wrap(0)) probe_clk_fb(clk_fb);
    
endmodule
