#!/usr/bin/env xmodelpy 
"""
SCRIPT jtran.py

It simulates a jitter transfer curve by sweeping frequency of an input sinusoidal jitter and measuring output phase response.

Revisions:
    $Id: jaeha@mics.snu.ac.kr-20140815004059-266qa7njgxq6ucrb $
    $Author: Jaeha Kim <jaeha@mics.snu.ac.kr> $
    $DateTime: Thu 2014-08-14 17:40:59 -0700 $

"""

##-------------------------------------------------------------------
## import modules
##-------------------------------------------------------------------

import sys
import numpy as np
import pylab
from xmulan import *


##-------------------------------------------------------------------
## helper functions
##-------------------------------------------------------------------

def option_parser():
    """ define usages and options of the command-lineinterface. """

    import optparse
    description = "jitter transfer curve simulation utility."
    usage = "%prog [options]"
    parser = optparse.OptionParser(description=description, usage=usage)

    # set options
    parser.add_option('-m', '--simulator', dest='simulator',
        help='simulator: can be chosen in [vcs, modelsim, ncverilog]',
        action='store', type='string', default=None)
    parser.add_option('-f', '--freq_ref', dest='ref_freq',
        help='frequency of reference pulse',
        action='store', type='float', default=125.0e6)
    parser.add_option('-t', '--t_lock', dest='t_lock',
        help='initial simulation period (PLL locking time)',
        action='store', type='float', default=1.0e-6)
    parser.add_option('-a', '--sj_amp', dest='sj_amp',
        help='amplitude of reference sinusoidal jitter',
        action='store', type='float', default=0.025)
    parser.add_option('-s', '--freq_max', dest='max_freq',
        help='maximum frequency of reference sinusoidal jitter',
        action='store', type='float', default=1.0e9)
    parser.add_option('-e', '--freq_min', dest='min_freq',
        help='minimum frequency of reference sinusoidal jitter',
        action='store', type='float', default=1.0e6)
    parser.add_option('-n', '--num_sweep', dest='num_sweep',
        help='number of frequency sweep of reference sinusoidal jitter',
        action='store', type='int', default=20)
    parser.add_option('-b', '--timescale', dest='timescale',
        help='Simulation time for each sampling offset',
        action='store', type='string', default='1ps/1ps')
    parser.add_option('-c', '--cmd', dest='cmd_file',
        help='command file location',
        action='store', type='string', default='../../cp_pll.f')
    parser.add_option('-r', '--source', dest='source',
        help='empy source file name',
        action='store', type='string', default='tb_jtran.sv.empy')
    parser.add_option('-v', '--var', dest='var',
        help='an xbit/bit variable to extract response',
       action='store', type='string', default='tb_jtran.clk_out')


    return parser


def compute_jtran(phase_wav, sj_freq, sj_amp, ref_freq, t_lock):
    """ compute the jitter transfer magnitude and phase """

    # assuming phase_wav is a piecewise-constant waveform
    phase_data = phase_wav
    phase_wav = phase_wav.get_markers()[1]
    if phase_wav[0] == phase_wav[1]: idx_start = 0
    else: idx_start = 1

    phase_temp = phase_wav[idx_start::2]/(2*np.pi)
    k=0    
    phase=[]
    for i in range(len(phase_temp)):
        if(i==0) : 
            phase.append(phase_temp[i])
        else :
            if(phase_temp[i]-phase_temp[i-1]>0.9) :
                k=k-1
            elif (phase_temp[i]-phase_temp[i-1]<-0.9):
                k=k+1
            else : 
                k=k
            phase.append(phase_temp[i]+1*k)


    #time = phase_wav.sweep[0][idx_start::2] 
    time = phase_data.get_markers()[0][idx_start::2] 
    C_basis = np.cos(2*np.pi*sj_freq*time)
    S_basis = np.sin(2*np.pi*sj_freq*time)
    C_corr = np.dot(phase, C_basis)*2/len(phase)
    S_corr = np.dot(phase, S_basis)*2/len(phase)

    tf_mag = np.sqrt(C_corr**2+S_corr**2) / sj_amp
    tf_phase = np.arctan2(-S_corr, C_corr)

    return tf_mag, tf_phase


##-------------------------------------------------------------------
## jitter transfer curve
##-------------------------------------------------------------------

def run():
    """ command-line interface. """

    # parse command-line arguments
    parser = option_parser()
    (opts,args) = parser.parse_args()

    # define sweep parameters
    M=mulan()
    M.new_sw(sj_amp=opts.sj_amp,        # sinusoidal jitter amplitude
             ref_freq=opts.ref_freq,    # reference clock frequency
             t_lock = opts.t_lock)      # PLL locking time

    sj_freq_list = np.logspace(np.log10(opts.max_freq), np.log10(opts.min_freq), opts.num_sweep)

    # list of source files
    sources = [opts.source]
    cmdfile = [opts.cmd_file]

    # perform simulation sweep
    for row in M.sw_iter(caching=True):
        jtran_mag = []
        jtran_phase = []

        for freq in sj_freq_list: 
            sim_time = '%dns' % ((row.t_lock + 10.0/freq)*1e9)
            sys.stdout.write("    f = %g MHz: " % (freq/1e6))
            sys.stdout.flush()

            runbase = row.xmodel(sources, cmdfile=cmdfile, top_cell='tb_jtran',
                    timescale=opts.timescale, sim_time=sim_time,
                    params=dict(sj_freq=freq), simulator=opts.simulator, prefix='SJfreq='+str(freq))
            row_sim = rowml().readmeas(runbase + "/xmodel.jez")

            phase_out = row_sim[opts.var+".phase"]
            mag, phase = compute_jtran(phase_out, freq, row.sj_amp, 
                                       row.ref_freq, row.t_lock)

            sys.stdout.write("|H(f)| = %g, <H(f) = %g\n" % (mag, phase))
            jtran_mag.append(20*np.log10(mag))
            jtran_phase.append(phase)

        row.frequency = sj_freq_list[::-1]
        row.jtran_mag = jtran_mag[::-1]
        row.jtran_phase = jtran_phase[::-1]

        import xwave
        import matplotlib.pyplot as plt

        window = xwave.open_xwave()
        page = window.create_page("Meas 1", "analysis")

        fig = plt.figure(figsize=(8.0, 4.0), dpi=96, facecolor="#FFFFFF")
        fig.subplots_adjust(left=0.1, right=0.95, top=0.95, bottom=0.15)
        ax = fig.add_subplot(111, axisbg="#FFFFFF")

        ax.semilogx(row.frequency, row.jtran_mag, '-o', 
                    linewidth=3.0,
                    markeredgewidth=3.0, 
                    markeredgecolor='b',
                    markersize=9.0,
                    markerfacecolor="#FFFFFF")
        
        plt.setp(ax.get_xticklabels(), fontsize=12)
        plt.setp(ax.get_yticklabels(), fontsize=12)
        plt.xlabel('Frequency [Hz]', fontsize=12)
        plt.ylabel('Magnitude [dB]', fontsize=12)
        
        page.add_newpane("Jitter Transfer Curve\n[%s]"%opts.var, fig)
        xwave.show_xwave()
        plt.savefig('jtran.png')

    # save results
    M.save('jtran.ml')

    # plot graph
    #M.plot('jtran_mag', sweep='frequency', plotfunc=pylab.semilogx,
    #       filename="jtran.png")


##----------------------------------------------------------------------
## command-line execution 
##----------------------------------------------------------------------

if __name__ == "__main__":
    run()

##----------------------------------------------------------------------
## end of jtran.py
##----------------------------------------------------------------------

