/*----------------------------------------------------------------------
MODULE digital_lf.v

= Purpose =

A digital loop filter for PLL.

= Description =

The digital loop filter has a basic PI-controller with an integral gain Ki
and a proportional gain Kp.

    out[n] = Kp*in[n] + Ki*sum(j=1..n-1) in[j]

= Parameters =

||'''NAME'''||'''TYPE'''||'''DESCRIPTION'''||'''UNIT'''||'''DEFAULT'''||
|| init_value || integer || initial control value || None || 10'b10_0000_0000 ||

= Revisions =

$Author$
$DateTIme$
$Id$
----------------------------------------------------------------------*/

module digital_lf (
    input [5:0] up,
    input [5:0] dn,
    input clk,                      // triggering clock
    input reset_n,                  // reset signal
    output reg [9:0] out,           // control output
    output reg signed [2:0] dsm_out,// dsm control output
    output reg clk_lf_dco           // loop-filter clock for the DCO retiming
);

    // variables
    wire    [9:0]  out_nxt;
    wire    [9:0]  in_ext;
    wire    [6:0]  in;

    // digital filter
    assign in = {1'b0,up} - {1'b0,dn};
    assign in_ext = {{3{in[6]}},in};

    always @(posedge clk or negedge reset_n) begin
        if (!reset_n) begin
            out <= 10'b10_0000_0000;    // initialization
            dsm_out <= 3'b0;
        end
        else begin
            out <= out_nxt;
            dsm_out <= 3'b0;
        end
    end

    assign out_nxt = out + in_ext;

    always @(clk) begin
        clk_lf_dco = clk;
    end

endmodule
