/*----------------------------------------------------------------------
MODEL ring_dco.sv

= Purpose =

A ring-based DCO block (custom analog).

= Description =

A fixed-timestep model.

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module ring_dco #(
    `parameter_real(K_dco, 0.00056),        // DCO relative gain
    `parameter_real(freq_center, 1.41e9),   // center of the DCO output frequency
    `parameter_real(PN_foffset, 1e6),       // phase noise frequency offset
    `parameter_real(PN_dbc, -85),           // phase noise at the frequency offset
    `parameter_real(delay_in_int, 512e-12), // integral path input delay
    `parameter_real(delay_in_prop, 121e-12),// proportional path input delay
    `parameter_real(delay_cq, 322e-12),     // clock-to-output delay
    `parameter_real(t_const_int, 224e-12),  // integral path time-constant
    `parameter_real(t_const_prop, 35e-12),  // proportional path time-constant
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 0.26e4)        // equivalent output resistance
)(
    input vdd,
    input vss,
    input clk_lf,               // input retiming clock
    input [2:0] dco_pgain,      // DCO proportional gain control (binary-coded)
    `input_xbit prop_up,        // proportional control (up, binary-coded)
    `input_xbit prop_dn,        // proportional control (dn, binary-coded)
    input [9:0] in,             // DCO control input (binary-coded)
    input signed [2:0] in_dsm,  // DCO control DSM input (binary-coded)
    input reset_n,	            // DCO start-up signal for current reference
    `output_xbit [7:0] clk_out  // uniformly-spaced multiphase output clock
);

    parameter PN_fcenter = freq_center;

    reg [9:0] in_delay;
    reg signed [2:0] in_dsm_delay;
    reg [30:0] int_coarse, int_fine;    // coarse & fine code (thermometer-coded)
    reg [2:0] int_dsm;                  // DSM code (binary-coded)
    reg [6:0] prop_gain;

    xbit [9:0] in_xbit, in_delay_xbit;
    xbit [2:0] in_dsm_xbit, in_dsm_delay_xbit;

    bit_to_xbit #(.width(10)) conn_in(.in(in), .out(in_xbit));
    bit_to_xbit #(.width(3)) conn_in_dsm(.in(in_dsm), .out(in_dsm_xbit));

    genvar j;
    generate 
        for (j=0; j<10; j=j+1) begin : gen_loop_in
        delay_xbit #(.delay(delay_in_int))
            delay_xbit_in(.in(in_xbit[j]), .out(in_delay_xbit[j]));
        end
        for (j=0; j<3; j=j+1) begin : gen_loop_in_dsm
        delay_xbit #(.delay(delay_in_int))
            delay_xbit_in_dsm(.in(in_dsm_xbit[j]), .out(in_dsm_delay_xbit[j]));
        end
    endgenerate
 
    xbit_to_bit #(.width(10)) conn_in_delay(.in(in_delay_xbit), .out(in_delay));
    xbit_to_bit #(.width(3)) conn_in_dsm_delay(.in(in_dsm_delay_xbit), .out(in_dsm_delay));
  
    // binary to thermometer encode with retimer
    ring_dco_encoder therm_encoder (
            .clk(clk_lf), .reset_n(reset_n),
            .in(in_delay), .in_dsm(in_dsm_delay), .dco_pgain(dco_pgain),
            .int_coarse(int_coarse), .int_fine(int_fine), .int_dsm(int_dsm),
            .prop_gain(prop_gain)
        );
    // core ring_dco instances
    ring_dco_core #(.num_phase(8),
        .int_coarse_steps(31), .int_fine_steps(31), .int_dsm_steps(3),
        .freq_center(freq_center), .K_dco(K_dco),
        .PN_fcenter(PN_fcenter), .PN_foffset(PN_foffset), .PN_dbc(PN_dbc),
        .delay_in_int(delay_cq), .delay_in_prop(delay_in_prop),
        .t_const_int(t_const_int), .t_const_prop(t_const_prop))
        ring_dco_core (.vdd(vdd), .vss(vss), .reset_n(reset_n),
                .prop_gain(prop_gain), .prop_up(prop_up), .prop_dn(prop_dn),
                .int_coarse(int_coarse), .int_fine(int_fine), .int_dsm(int_dsm),
                .out(clk_out));

endmodule

module ring_dco_bin_encoder_dsm( 
    input [2:0] din,        // binary-coded input
    output reg [2:0] dout   // thermometer-coded output
);

    // binary-to-thermometer encoder
    always @(din) begin
        case(din)
            3'b100 : dout = 3'b111;
            3'b101 : dout = 3'b111;
            3'b110 : dout = 3'b110;
            3'b111 : dout = 3'b101;
            3'b000 : dout = 3'b000;
            3'b001 : dout = 3'b001;
            3'b010 : dout = 3'b010;
            3'b011 : dout = 3'b011;
        endcase
    end

endmodule

module ring_dco_core #(
    `parameter_integer(num_phase, 8),           // number of output phases
    `parameter_integer(int_coarse_steps, 31),   // integral control bits (thermometer/coarse)
    `parameter_integer(int_fine_steps, 31),     // integral control bits (thermometer/fine)
    `parameter_integer(int_dsm_steps, 7),       // integral control bits (thermometer/DSM)
    `parameter_real(K_dco, 0.00056),        // DCO relative gain
    `parameter_real(freq_center, 1.5e9),    // center of the DCO output frequency
    `parameter_real(init_phase, M_PI/2),    // initial phase (randomized if <0)
    `parameter_real(RJ_kappa, 0.0e-7),      // RMS random jitter after 1-second free-running
    `parameter_real(PN_fcenter, 1.5e9),     // phase noise center frequency
    `parameter_real(PN_foffset, 1.0e6),     // phase noise offset frequency
    `parameter_real(PN_dbc, -85),           // phase noise measured in dBc
    `parameter_real(delay_in_int, 150.0e-12),   // integral path input delay
    `parameter_real(delay_in_prop, 200.0e-12),  // proportional path input delay
    `parameter_real(t_const_int, 100.0e-12),    // integral path time-constant
    `parameter_real(t_const_prop, 40.0e-12)     // proportional path time-constant
)(
    input vdd,
    input vss,
    input reset_n,                          // reset (active-low)
    `input_xbit prop_up,                    // proportional up input pulse bit
    `input_xbit prop_dn,                    // proportional down input pulse bit
    input [int_coarse_steps-1:0] int_coarse,// integral input code (thermometer/coarse)
    input [int_fine_steps-1:0] int_fine,    // integral input code (thermometer/fine)
    input [int_dsm_steps-1:0] int_dsm,      // DSM input code (thermometer-coded)
    input [6:0] prop_gain,                  // DCO proportional gain control (thermometer-coded)
    `output_xbit [num_phase-1:0] out        // output clocks
);

    // parameters
    parameter real BW_int = 1/t_const_int;
    parameter real BW_prop = 1/t_const_prop;

    // variables
    real prop_gain_r;
    real freq;
    xreal freq_xr;

    bit [num_phase-1:0] out_init;
    xbit [num_phase-1:0] out_init_xbit;
    xbit [num_phase-1:0] out_xbit;
    xbit rstn;

    xreal int_ctrl_xr, dsm_ctrl_xr, prop_up_xr, prop_dn_xr;
    xreal int_ctrl_filter, dsm_ctrl_filter, prop_up_filter, prop_dn_filter;
    xreal int_factor, prop_factor;
    real int_factor_r, prop_factor_r;

    integer i;

    // DCO proportional gain control
    initial begin
        for(i=0; i<num_phase; i=i+1) begin
            if(((i+1)/2 - i/2) == 1) out_init[i] = 1'b1;
            else out_init[i] = 1'b0;
        end
    end

    bit_to_xbit #(.width(num_phase)) conn0(.in(out_init), .out(out_init_xbit));

    always @(prop_gain) begin
        case (prop_gain)
            7'b0000001 : prop_gain_r = 0.5*32;
            7'b0000011 : prop_gain_r = 1*32;
            7'b0000111 : prop_gain_r = 2*32;
            7'b0001111 : prop_gain_r = 4*32;
            7'b0011111 : prop_gain_r = 8*32;
            7'b0111111 : prop_gain_r = 16*32;
            7'b1111111 : prop_gain_r = 32*32;
            default : prop_gain_r = 0*32;
        endcase
    end

    real int_ctrl, dsm_ctrl;

    always @(int_coarse or int_fine or int_dsm) begin
        // integral contrl
        int_ctrl = 0.0;
        dsm_ctrl = 0.0;
        for (i = 0; i < int_coarse_steps; i=i+1)
            int_ctrl = int_ctrl+int_coarse[i];
        int_ctrl = int_ctrl*(int_coarse_steps + 1);

        for (i = 0; i < int_fine_steps; i=i+1)
            int_ctrl = int_ctrl+int_fine[i];

        if(int_dsm[int_dsm_steps-1])
            dsm_ctrl = -real'(int_dsm[int_dsm_steps-2:0]);
        else
            dsm_ctrl = real'(int_dsm[int_dsm_steps-2:0]);
    end

    real_to_xreal conn_int(.in(int_ctrl), .out(int_ctrl_xr));
    real_to_xreal conn_dsm(.in(dsm_ctrl), .out(dsm_ctrl_xr));
    transition tran_pup(.in(prop_up), .out(prop_up_xr));
    transition tran_pdn(.in(prop_dn), .out(prop_dn_xr));

    filter #(.poles('{BW_int,0.0}), .zeros('{0}), .delay(delay_in_int))
            filter_int(.in(int_ctrl_xr), .out(int_ctrl_filter));

    filter #(.poles('{BW_int,0.0}), .zeros('{0}), .delay(delay_in_int))
            filter_dsm(.in(dsm_ctrl_xr), .out(dsm_ctrl_filter));

    filter_var #(.delay(delay_in_prop), .num_poles(1), .num_zeros(0))
            filter_prop_up(.in(prop_up_xr), .out(prop_up_filter),
                           .poles('{BW_prop,0.0}), .zeros(),
                           .gain(prop_gain_r));
    filter_var #(.delay(delay_in_prop), .num_poles(1), .num_zeros(0))
            filter_prop_dn(.in(prop_dn_xr), .out(prop_dn_filter),
                           .poles('{BW_prop,0.0}), .zeros(),
                           .gain(prop_gain_r));

    poly_func #(.num_in(2), .data('{-(int_coarse_steps+1)*(int_fine_steps+1)/2.0,1.0,1.0})) poly_func0(.in({int_ctrl_filter,dsm_ctrl_filter}), .out(int_factor));
    add #(.scale('{K_dco,-K_dco})) add0(.in({prop_up_filter,prop_dn_filter}), .out(prop_factor));

    xreal_to_real conn_int_factor(.in(int_factor), .out(int_factor_r));
    xreal_to_real conn_prop_factor(.in(prop_factor), .out(prop_factor_r));

    // digital-to-frequency conversion (exponential D-to-f)
    always @(int_factor_r or prop_factor_r) begin
        freq = freq_center * `pow(1 + K_dco, int_factor_r) * (1 + prop_factor_r);
    end

    real_to_xreal conn1(.in(freq), .out(freq_xr));

    // frequency-to-clock conversion
    freq_to_clk #(.init_phase(init_phase), .num_phase(num_phase),
                .RJ_kappa(RJ_kappa), .PN_foffset(PN_foffset),
                .PN_fcenter(PN_fcenter), .PN_dbc(PN_dbc))
            freq2clk(.in(freq_xr), .out(out_xbit));

    bit_to_xbit conn2(.in(reset_n), .out(rstn));

    genvar j;
    generate
        for (j=0; j<num_phase; j=j+1) begin : gen_loop
        mux_xbit out_gen(.in({out_xbit[j],out_init_xbit[j]}), .sel(rstn), .out(out[j]));
        end
    endgenerate

endmodule

module ring_dco_encoder(
    input clk,                  // input retiming clock
    input reset_n,              // reset (active-low)
    input [9:0] in,             // binary-coded integral path input
    input signed [2:0] in_dsm,  // binary-coded DSM input (signed)
    input [2:0] dco_pgain,      // bianry-coded proportional path gain control code
    output reg [30:0] int_coarse,   // thermometer-coded integral code (coarse)
    output reg [30:0] int_fine,     // thermometer-coded integral code (fine)
    output reg [2:0] int_dsm,       // binary-coded integral code (DSM)
    output reg [6:0] prop_gain      // thermometer-ocded proportional path gain control code
);

    reg [9:0] in_retimed;
    reg [2:0] in_dsm_retimed;
    reg [2:0] dco_pgain_retimed;

    // binary to thermometer decoder
    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            in_retimed <= 10'b10_0000_0000;
            in_dsm_retimed <= 3'b000;
            dco_pgain_retimed <= 3'b000;
        end
        else begin
            in_retimed <= in;
            in_dsm_retimed <= in_dsm;
            dco_pgain_retimed <= dco_pgain;
        end
    end

    ring_dco_therm_encoder therm_coarse(.din(in_retimed[9:5]), .dout(int_coarse));
    ring_dco_therm_encoder therm_fine(.din(in_retimed[4:0]), .dout(int_fine));

    ring_dco_bin_encoder_dsm bin_dsm(.din(in_dsm_retimed), .dout(int_dsm));
    ring_dco_therm_encoder_pgain therm_pgain(.din(dco_pgain_retimed), .dout(prop_gain));

endmodule

module ring_dco_therm_encoder( 
    input [4:0] din,        // binary-coded input
    output reg [30:0] dout  // thermometer-coded output
);

    // binary-to-thermometer encoder
    always @(din) begin
        case(din)
            5'b00000 : dout = 31'b0000000000000000000000000000000;
            5'b00001 : dout = 31'b0000000000000000000000000000001;
            5'b00010 : dout = 31'b0000000000000000000000000000011;
            5'b00011 : dout = 31'b0000000000000000000000000000111;
            5'b00100 : dout = 31'b0000000000000000000000000001111;
            5'b00101 : dout = 31'b0000000000000000000000000011111;
            5'b00110 : dout = 31'b0000000000000000000000000111111;
            5'b00111 : dout = 31'b0000000000000000000000001111111;

            5'b01000 : dout = 31'b0000000000000000000000011111111;
            5'b01001 : dout = 31'b0000000000000000000000111111111;
            5'b01010 : dout = 31'b0000000000000000000001111111111;
            5'b01011 : dout = 31'b0000000000000000000011111111111;
            5'b01100 : dout = 31'b0000000000000000000111111111111;
            5'b01101 : dout = 31'b0000000000000000001111111111111;
            5'b01110 : dout = 31'b0000000000000000011111111111111;
            5'b01111 : dout = 31'b0000000000000000111111111111111;

            5'b10000 : dout = 31'b0000000000000001111111111111111;
            5'b10001 : dout = 31'b0000000000000011111111111111111;
            5'b10010 : dout = 31'b0000000000000111111111111111111;
            5'b10011 : dout = 31'b0000000000001111111111111111111;
            5'b10100 : dout = 31'b0000000000011111111111111111111;
            5'b10101 : dout = 31'b0000000000111111111111111111111;
            5'b10110 : dout = 31'b0000000001111111111111111111111;
            5'b10111 : dout = 31'b0000000011111111111111111111111;

            5'b11000 : dout = 31'b0000000111111111111111111111111;
            5'b11001 : dout = 31'b0000001111111111111111111111111;
            5'b11010 : dout = 31'b0000011111111111111111111111111;
            5'b11011 : dout = 31'b0000111111111111111111111111111;
            5'b11100 : dout = 31'b0001111111111111111111111111111;
            5'b11101 : dout = 31'b0011111111111111111111111111111;
            5'b11110 : dout = 31'b0111111111111111111111111111111;
            5'b11111 : dout = 31'b1111111111111111111111111111111;
        endcase
    end

endmodule

module ring_dco_therm_encoder_pgain( 
    input [2:0] din,        // binary-coded input
    output reg [6:0] dout   // thermometer-coded output
);

    // binary-to-thermometer encoder
    always @(din) begin
        case(din)
            3'b000 : dout = 7'b0000000;
            3'b001 : dout = 7'b0000001;
            3'b010 : dout = 7'b0000011;
            3'b011 : dout = 7'b0000111;
            3'b100 : dout = 7'b0001111;
            3'b101 : dout = 7'b0011111;
            3'b110 : dout = 7'b0111111;
            3'b111 : dout = 7'b1111111;
        endcase
    end

endmodule
