/*----------------------------------------------------------------------
TESTBENCH tb_ring_dco.sv

= Purpose =

A testbench to measure the digital-to-frequency characteristic of the dco_block module.

= Description =

The testbench sweeps the input codes and checks the output frequencies and phases.

= Revisions =

$Author$
$DateTIme$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module tb_ring_dco();

    reg [2:0] dco_pgain = 3'b000;
    reg [9:0] in = 10'b00_0000_0000;
    reg [2:0] in_dsm = 3'b000;
    reg       reset_n;

    reg clk;

    xbit prop_up, prop_dn;
    xbit [7:0] clk_out;

    xbit clk_xbit;
    xreal period_xr;
    real period;

    reg vdd = 1;
    reg vss = 0;

    // stimuli
    const_xbit #(.width(2), .value(2'b00)) const_xbit0(.out({prop_up,prop_dn}));

    initial begin
        reset_n = 1'b1;
        #(1e-9/`TIME_SCALE) reset_n = 1'b0;
        #(2e-9/`TIME_SCALE) reset_n = 1'b1;
    end

    clk_gen #(.freq(50.0e6)) clk_gen(.out(clk_xbit));
    xbit_to_bit conn_clk(.in(clk_xbit), .out(clk));

    // DUT
    ring_dco DUT(
        .vdd(vdd),
        .vss(vss),
        .clk_lf(clk),
        .dco_pgain(dco_pgain),
        .prop_up(prop_up),
        .prop_dn(prop_dn),
        .in(in),
        .in_dsm(in_dsm),
        .reset_n(reset_n),
        .clk_out(clk_out)
    );

    // test sequence
    reg mode;
    reg flag_rgain;

    initial begin
        mode = 1;           // 0: quiet, 1: verbose
        flag_rgain = 1;     // 0: fail, 1: pass
        #(100e-9/`TIME_SCALE)
        meas_rgain;
    end

    clk_to_period clk2period(.in(clk_out[0]), .out(period_xr));
    xreal_to_real conn_period(.in(period_xr), .out(period));

    task meas_rgain;
        real fstep;
        real fstep_avg = 0.0;
        real f_dco_prev = -1;
        real f_dco_real;
        real t_dco_real;
        integer i, j;
        integer f;

        f = $fopen("result.txt","w");
        $fwrite(f,"CODE\t\tFREQ\n");
        for(i=0; i<1024; i++) begin
            in = i;
            t_dco_real = 0;
            @(posedge clk);
            @(posedge clk);
            for(j=0; j<10; j++) begin
                @(posedge `value(clk_out[0]));
                t_dco_real = t_dco_real + period;
            end
            f_dco_real = 10/t_dco_real;
            if (mode) begin
                $write("   in=%b: f_dco = %g\n", in, f_dco_real);
                $fwrite(f,"%d\t%g\n", in, f_dco_real);
            end

            if (f_dco_prev > 0) begin
                fstep = f_dco_real / f_dco_prev;
                fstep_avg += `log(fstep);
            end

            f_dco_prev = f_dco_real;
        end

        fstep_avg = `pow(M_E, fstep_avg/1023);
        $write("------------------------------------------------------------------------------\n");
        $write("DCO RGAIN TEST: avg = %g\n", fstep_avg - 1); 
        $write("------------------------------------------------------------------------------\n");

        $finish;
    endtask

    real in_r;
    always @(in) begin
        in_r = real'(in);
    end

    // probing
    probe_xbit  probe_clk0(clk_out[0]);
    probe_xbit  probe_clk1(clk_out[1]);
    probe_xbit  probe_clk2(clk_out[2]);
    probe_xbit  probe_clk3(clk_out[3]);
    probe_xbit  probe_clk4(clk_out[4]);
    probe_xbit  probe_clk5(clk_out[5]);
    probe_xbit  probe_clk6(clk_out[6]);
    probe_xbit  probe_clk7(clk_out[7]);
    probe_freq  probe_freq_clk_out(clk_out[0]);
    probe_real  probe_in_r(in_r);

endmodule
