/*----------------------------------------------------------------------
MODEL ph_rotator.sv

= Purpose =

A phase rotator block.

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

module ph_rotator #(
    `parameter_real(f_range_cof_0 [0:2], '{-0.963,21.0,0.470}),  // frequency tuning coefficients for f_range=0
    `parameter_real(f_range_cof_1 [0:2], '{141,400,6.38}),       // frequency tuning coefficients for f_range=1
    `parameter_real(f_range_cof_2 [0:2], '{-934,6780,29.2}),     // frequency tuning coefficients for f_range=2
    `parameter_real(f_range_cof_3 [0:2], '{148e3,110e3,-268}),   // frequency tuning coefficients for f_range=3
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 0.26e3)        // equivalent output resistance
)(
    input vdd,
    input vss,
    input reset_n,              // reset (active low)
    input [1:0] f_range,        // operating frequency range (global)
    `input_xbit [7:0] clk_in,   // multiphase clock inputs
    input [6:0] phsel,          // phase select input
    `output_xbit clk_out,       // clock output
    `output_xbit clk_outb       // clock output (negative)
);

    wire pulse;
    reg clk_fdc;
    reg [5:0] f_bias, fdc_ctrl;

    // PH_ROTATOR core
    ph_rotator_core #(.f_range_cof_0(f_range_cof_0),
                      .f_range_cof_1(f_range_cof_1),
                      .f_range_cof_2(f_range_cof_2),
                      .f_range_cof_3(f_range_cof_3))
        ph_rotator_core (
            .vdd(vdd), .vss(vss),
            .reset_n(reset_n),
            .f_range(f_range), .f_bias(f_bias), .fdc_ctrl(fdc_ctrl),
            .clk_in(clk_in), .phsel(phsel),
            .clk_out(clk_out), .clk_outb(clk_outb),
            .clk_fdc(clk_fdc), .pulse(pulse)
        );

    // FDC logic
    ph_fdc_logic ph_fdc_logic (
        .clk(clk_fdc), .reset_n(reset_n), .pulse(pulse),
        .dctrl(fdc_ctrl), .out(f_bias));

endmodule

module ph_fdc_acc (
    input clk,
    input reset_n,
    input rst_sync_global,
    input rst_sync_local,
    input enable,
    input [4:0] in,
    output reg [3:0] n_up,
    output reg [3:0] n_dn
);

    reg [3:0] n_up_nxt, n_dn_nxt;
    reg [1:0] flag;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) flag <= 2'b0;
        else begin
            if(rst_sync_local) begin
                flag <= 2'b0;
            end
            else begin
                if(in == 5'b01110) begin     // 15 - 1
                    if(flag[1]) begin
                        flag <= 2'b11;
                    end
                    else begin
                        flag <= 2'b01;
                    end
                end
                else begin
                    if(in == 5'b01111) begin // 15+1 - 1
                        flag <= 2'b11;
                    end
                    else begin
                        flag <= flag;
                    end
                end
            end
        end
    end
    
    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            n_up <= 4'b0;
            n_dn <= 4'b0;
        end
        else begin
            if(rst_sync_global) begin
                n_up <= 4'b0;
                n_dn <= 4'b0;
            end
            else begin
                if(enable) begin
                    n_up <= n_up_nxt;
                    n_dn <= n_dn_nxt;
                end
                else begin
                    n_up <= n_up;
                    n_dn <= n_dn;
                end
            end
        end
    end

    always @(*) begin
        case(flag)
            2'b00 :
                begin
                    n_up_nxt = n_up;
                    n_dn_nxt = n_dn + 1;
                end
            2'b11 :
                begin
                    n_up_nxt = n_up + 1;
                    n_dn_nxt = n_dn;
                end
            default :
                begin
                    n_up_nxt = n_up;
                    n_dn_nxt = n_dn;
                end
        endcase
    end

endmodule

module ph_fdc_controller (
    input [1:0] f_range,        // operation frequency range
    input clk_in,               // clock whose frequency is going to be acquired
    input clk_osc,              // oscillator operating frequency
    input reset_n,              // reset (active low)
    output reg clk_fdc,         // clock for fdc_logic
    output reg pulse            // pulse signal to operate fdc_logic
);

    reg clk_osc_div, clk_in1, clk_osc1, clk_osc_div2;
    wire clk_in_mux, clk_osc_mux;

    assign clk_in_mux = (f_range == 2'b11) ? clk_in1 : clk_in;
    assign clk_osc_mux = (f_range == 2'b11) ? clk_osc1 : clk_osc_div2;

    always @(clk_in_mux) begin
        clk_fdc = clk_in_mux;
    end

    ph_fdc_div #(.div_factor(2))
            fdc_div2_clk_osc(.clk(clk_osc), .reset_n(reset_n),
                             .clk_div(clk_osc_div2));

    ph_fdc_div #(.div_factor(8))
            fdc_div8_clk_in(.clk(clk_in), .reset_n(reset_n),
                            .clk_div(clk_in1));
    ph_fdc_div #(.div_factor(8))
            fdc_div8_clk_osc(.clk(clk_osc_div2), .reset_n(reset_n),
                             .clk_div(clk_osc1));

    ph_fdc_div #(.div_factor(16))
            fdc_div(.clk(clk_osc_mux), .reset_n(reset_n), .clk_div(clk_osc_div));

    ph_fdc_pulse_gen fdc_pulse_gen(.clk(clk_fdc), .in(clk_osc_div), .reset_n(reset_n), .pulse(pulse));

endmodule

module ph_fdc_counter (
    input clk,
    input reset_n,
    input rst_sync,
    output reg [4:0] cnt
);

    reg [4:0] cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 5'b0;
        end
        else begin
            if(rst_sync) begin
                cnt <= 5'b0;
            end
            else begin
                cnt <= cnt_nxt;
            end
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module ph_fdc_counter_en (
    input clk,
    input reset_n,
    input enable,
    output reg [3:0] cnt
);

    reg [3:0] cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 4'b0;
        end
        else begin
            if(enable)
                if(cnt == 4'b1110) cnt <= 4'b0;
                else cnt <= cnt_nxt;
            else
                cnt <= cnt;
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module ph_fdc_decision (
    input clk,
    input reset_n,
    input enable,
    input enable_global,
    input [3:0] n_up,
    input [3:0] n_dn,
    output reg [5:0] interior_code,
    output reg [5:0] out
);

    wire change;
    reg n_up_msb_pre, n_dn_msb_pre;
    reg [5:0] interior_code_nxt, exterior_code_nxt, history;

    assign change = ((n_up[3]^n_up_msb_pre) | (n_dn[3]^n_dn_msb_pre)) ? 1'b1 : 1'b0;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            interior_code <= 6'b100100;
            out <= 6'b100100;
        end
        else begin
            out <= exterior_code_nxt;
            if(enable) interior_code <= interior_code_nxt;
            else interior_code <= interior_code;
        end
    end

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            n_up_msb_pre <= 1'b0;
            n_dn_msb_pre <= 1'b0;
        end
        else begin
            if(enable_global) begin
                n_up_msb_pre <= n_up[3];
                n_dn_msb_pre <= n_dn[3];
            end
            else begin
                n_up_msb_pre <= n_up_msb_pre;
                n_dn_msb_pre <= n_dn_msb_pre;
            end
        end
    end

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            history <= 6'b0;
        end
        else begin
            if(enable_global) begin
                if(change) history <= {history[5:0],1'b1};
                else history <= {history[5:0],1'b0};
            end
            else begin
                history <= history;
            end
        end
    end

    always @(*) begin
        if(n_up[3]) begin
            if(interior_code == 6'b111111) interior_code_nxt = 6'b111111;
            else interior_code_nxt = interior_code + 1;
        end
        else if(n_dn[3]) begin
            if(interior_code == 6'b000000) interior_code_nxt = 6'b000000;
            else interior_code_nxt = interior_code - 1;
        end
        else begin
            interior_code_nxt = interior_code;
        end
    end

    always @(*) begin
        if(&history) exterior_code_nxt = out;
        else exterior_code_nxt = interior_code;
    end

endmodule

module ph_fdc_div #(
    `parameter_integer(div_factor, 8)
)(
    input clk,
    input reset_n,
    output reg clk_div
);

    reg [9:0] cnt, cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 0;
            clk_div <= 0;
        end
        else begin
            if(cnt == div_factor/2-1) begin
                cnt <= 0;
                clk_div <= ~clk_div;
            end
            else begin
                cnt <= cnt_nxt;
            end
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module ph_fdc_logic (
    input clk,                  // clock whose frequency is going to be acquired
    input reset_n,              // reset (active low)
    input pulse,                // pulse detecting the rising edge of the divided clk_osc
    output reg [5:0] dctrl,     // control code only in FDC
    output reg [5:0] out        // control code output
);

    wire en_cnt, en_acc, en_dec, en_dec_global, rst_cnt, rst_acc_global, rst_acc_local;
    reg period, period_retimed, pulse_acc, pulse_retimed;
    reg [4:0] cnt;
    reg [3:0] n_up, n_dn, cnt_period;

    assign en_cnt = pulse;
    assign en_acc = pulse;
    assign en_dec = period;
    assign en_dec_global = pulse_acc;
    assign rst_cnt = pulse;
    assign rst_acc_global = pulse_acc;
    assign rst_acc_local = pulse_retimed;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            pulse_retimed <= 1'b0;
            period_retimed <= 1'b0;
        end
        else begin
            pulse_retimed <= pulse;
            period_retimed <= period;
        end
    end

    always @(*) begin
        pulse_acc = period&(~period_retimed);
        if(cnt_period == 4'b1110) period = 1'b1;
        else period = 1'b0;
    end 

    ph_fdc_counter fdc_counter_fd(.clk(clk), .reset_n(reset_n), .rst_sync(rst_cnt), .cnt(cnt));

    ph_fdc_counter_en fdc_counter_acc(.clk(clk), .reset_n(reset_n), .enable(en_cnt), .cnt(cnt_period));

    ph_fdc_acc fdc_acc(.clk(clk), .reset_n(reset_n), .rst_sync_global(rst_acc_global), .rst_sync_local(rst_acc_local), .enable(en_acc), .in(cnt), .n_up(n_up), .n_dn(n_dn));

    ph_fdc_decision fdc_decision(.clk(clk), .reset_n(reset_n), .enable(en_dec), .enable_global(en_dec_global), .n_up(n_up), .n_dn(n_dn), .interior_code(dctrl), .out(out));

endmodule

module ph_fdc_pulse_gen (
    input clk,
    input in,
    input reset_n,
    output reg pulse
);

    reg reg1, reg2;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            reg1 <= 1'b0;
            reg2 <= 1'b0;
        end
        else begin
            reg1 <= in;
            reg2 <= reg1;
        end
    end

    always @(*) begin
        pulse = reg1&(~reg2);
    end

endmodule

module ph_interp #(
    `parameter_real(f_range_cof_0 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=0
    `parameter_real(f_range_cof_1 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=1
    `parameter_real(f_range_cof_2 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=2
    `parameter_real(f_range_cof_3 [0:2], '{0.00,0.00,0.00})  // frequency tuning coefficients for f_range=3
)(
    input vdd,
    input vss,
    input [1:0] f_range,        // operating frequency range (global)
    input [5:0] f_bias,         // bias corresponding to operating frequency
    `input_xbit clk0,			// clock input (selected when weight=0)
    `input_xbit clk1,			// clock input (selected when weight=1)
    `output_xbit clk_out,       // clock output (pos)
    `output_xbit clk_outb,	    // clock output (neg)
    input [3:0] weight,		    // thermometer-coded weight (pos)
    input [3:0] weight_b,	    // thermometer-coded weight (neg)
    input reset_n               // reset (active low)
);

    parameter real scale_factor = 1.7;
    parameter real cap = 1.0e-14;
    parameter real vth = 0.5;

    real freq;
    real delay;
    real w, wb;
    real ictrl0, ictrl1;

    xreal x1, vc1, x2, vc2, ictrl0_xr, ictrl1_xr;
    xbit qb, set, rst, pwmb, set_ex, set_ex2, sr, srb, rst_ex, rst_ex2, sr_ex, rstb;
    xbit clk0_b, clk1_b;

    always @(f_range or f_bias) begin
        case(f_range)
            2'b00 : freq = `pow(f_range_cof_0[0] + f_range_cof_0[1] * real'(f_bias) + f_range_cof_0[2] * real'(f_bias) * real'(f_bias), 0.5) * 1e6;
            2'b01 : freq = `pow(f_range_cof_1[0] + f_range_cof_1[1] * real'(f_bias) + f_range_cof_1[2] * real'(f_bias) * real'(f_bias), 0.5) * 1e6;
            2'b10 : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1] * real'(f_bias) + f_range_cof_2[2] * real'(f_bias) * real'(f_bias), 0.5) * 1e6;
            2'b11 : freq = `pow(f_range_cof_3[0] + f_range_cof_3[1] * real'(f_bias) + f_range_cof_3[2] * real'(f_bias) * real'(f_bias), 0.5) * 1e6;
            default : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1] * real'(f_bias) + f_range_cof_2[2] * real'(f_bias) * real'(f_bias), 0.5) * 1e6;
        endcase
    end

    // interpolator delay calculation
    initial begin
        case(f_range)
            2'b00 : delay = 1/`pow(f_range_cof_0[0] + f_range_cof_0[1]*63.0 + f_range_cof_0[2]*63.0*63.0, 0.5)/1e6/8*scale_factor;
            2'b01 : delay = 1/`pow(f_range_cof_1[0] + f_range_cof_1[1]*63.0 + f_range_cof_1[2]*63.0*63.0, 0.5)/1e6/8*scale_factor;
            2'b10 : delay = 1/`pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5)/1e6/8*scale_factor;
            2'b11 : delay = 1/`pow(f_range_cof_3[0] + f_range_cof_3[1]*63.0 + f_range_cof_3[2]*63.0*63.0, 0.5)/1e6/8*scale_factor;
            default : delay = 1/`pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5)/1e6/8*scale_factor;
        endcase
    end

    always @(freq) begin
        delay = 1/freq/8*scale_factor;
    end

    always @(*) begin
        ictrl0 = cap/delay * wb * vth;
        ictrl1 = cap/delay * w * vth;
    end

    // weight calculation
    always @(*) begin
        w = (1.0*(8*real'(weight[3])+4*real'(weight[2])+2*real'(weight[1])+real'(weight[0]))+0.5)/16.0;
        wb = (1.0*(8*real'(weight_b[3])+4*real'(weight_b[2])+2*real'(weight_b[1])+real'(weight_b[0]))+0.5)/16.0;
    end

    // reset
    bit_to_xbit conn_b2xb0(.in(reset_n), .out(rstb));

    // phase interpolation
    real_to_xreal r2xr0(.in(ictrl0), .out(ictrl0_xr));
    real_to_xreal r2xr1(.in(ictrl1), .out(ictrl1_xr));

    inv_xbit inv_clk0(.in(clk0), .out(clk0_b));
    inv_xbit inv_clk1(.in(clk1), .out(clk1_b));

    isource #(.mode("in"))
            i1(.pos(`ground),.neg(x1), .in(ictrl0_xr));
    switch #(.R1(1e7), .R0(1e-3), .ic(0)) 
            sw_p1(.pos(x1), .neg(vc1), .ctrl(clk0));
    switch #(.R1(1e7), .R0(1e-3), .ic(0)) 
            sw_p1b(.pos(x2), .neg(vc1), .ctrl(clk1));
    switch #(.R1(1e-3), .R0(1e7), .ic(0))
            sw_n1(.pos(vc1), .neg(`ground), .ctrl(sr));

    capacitor #(.C(cap), .ic(0)) cap1(.pos(vc1), .neg(`ground));
    slice #(.threshold(vth)) slice1(.in(vc1), .in_ref(`ground), .out(set_ex));
    inv_xbit inv_set(.in(set_ex), .out(set_ex2));
    and2_xbit and2_set(.in_a(set_ex2),.in_b(rstb),.out(set));
    
    isource #(.mode("in"))
            i2(.pos(`ground),.neg(x2), .in(ictrl1_xr));
    switch #(.R1(1e7), .R0(1e-3), .ic(0))
            sw_p2(.pos(x1), .neg(vc2), .ctrl(clk0_b));
    switch #(.R1(1e7), .R0(1e-3), .ic(0))
            sw_p2b(.pos(x2), .neg(vc2), .ctrl(clk1_b));
    switch #(.R1(1e-3), .R0(1e7), .ic(0))
            sw_n2(.pos(vc2), .neg(`ground), .ctrl(srb));
    capacitor #(.C(cap), .ic(0)) cap2(.pos(vc2), .neg(`ground));

    slice #(.threshold(vth)) slice2(.in(vc2), .in_ref(`ground), .out(rst_ex));
    buf_xbit buf_rst(.in(rst_ex), .out(rst_ex2));
    nand2_xbit nand2_rst(.in_a(rst_ex2),.in_b(rstb),.out(rst));

    // SR latch
    nand2_xbit nand_rslatch0(.in_a(set), .in_b(srb), .out(sr_ex));
    nand2_xbit nand_rslatch1(.in_a(rst), .in_b(sr_ex), .out(srb));

    inv_xbit inv1(.in(sr),.out(clk_out));
    inv_xbit inv2(.in(srb),.out(clk_outb));
    mux_xbit mux1(.in({sr_ex,`one_xbit}),.sel(rstb),.out(sr));

endmodule

module ph_mixer #(
    `parameter_real(f_range_cof_0 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=0
    `parameter_real(f_range_cof_1 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=1
    `parameter_real(f_range_cof_2 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=2
    `parameter_real(f_range_cof_3 [0:2], '{0.00,0.00,0.00})  // frequency tuning coefficients for f_range=3
)(
    input vdd,
    input vss,
    input reset_n,              // reset (active low)
    input [1:0] f_range,        // operating frequency range (globa)
    input [5:0] f_bias,         // bias corresponding to operating frequency
    `input_xbit [7:0] clk_in,   // multiphase clock inputs
    input [3:0] sel_even,       // phase mux selection inputs (even)
    input [3:0] sel_odd,        // phase mux selection inputs (odd)
    input [3:0] weight,         // phase interpolator weight control
    `output_xbit piclk,         // interpolated clock output (pos)
    `output_xbit piclkb         // interpolated clock output (neg)
);

    wire [3:0] weightb;
    xbit [1:0] clk_mux;
    xbit [3:0] sel_even_xbit;
    xbit [3:0] sel_odd_xbit;

    assign weightb = ~weight;

    // phase multiplexers
    bit_to_xbit #(.width(4)) conn_even(.in(sel_even), .out(sel_even_xbit));
    bit_to_xbit #(.width(4)) conn_odd(.in(sel_odd), .out(sel_odd_xbit));

    ph_mux ph_mux_even (
        .in({clk_in[6],clk_in[4],clk_in[2],clk_in[0]}), .out(clk_mux[0]), .sel(sel_even_xbit)
    );
    ph_mux ph_mux_odd (
        .in({clk_in[7],clk_in[5],clk_in[3],clk_in[1]}), .out(clk_mux[1]), .sel(sel_odd_xbit)
    );

    // phase interpolator
    ph_interp #(.f_range_cof_0(f_range_cof_0),
                .f_range_cof_1(f_range_cof_1),
                .f_range_cof_2(f_range_cof_2),
                .f_range_cof_3(f_range_cof_3))
        ph_interp(.vdd(vdd), .vss(vss),
                  .f_range(f_range), .f_bias(f_bias),
                  .clk0(clk_mux[0]), .clk1(clk_mux[1]),
                  .clk_out(piclk), .clk_outb(piclkb),
                  .weight(weight), .weight_b(weightb),
                  .reset_n(reset_n)
    );

endmodule

module ph_mux (
    `input_xbit [3:0] in,   // clock inputs
    `output_xbit out,       // clock output
    `input_xbit [3:0] sel   // selection control (one-hot encoded)
);

    // 4:1 multiplexer
    tribuf_xbit tribuf0(.en(sel[0]), .in(in[0]), .out(out));
    tribuf_xbit tribuf1(.en(sel[1]), .in(in[1]), .out(out));
    tribuf_xbit tribuf2(.en(sel[2]), .in(in[2]), .out(out));
    tribuf_xbit tribuf3(.en(sel[3]), .in(in[3]), .out(out));

endmodule

module ph_mux_enc (
    input vdd,
    input vss,
    input clk,                  // clock input (positive)
    input clkb,                 // clock input (negative)
    input reset_n,              // reset (active low)
    input [6:0] sel_in,         // mux selection signal inputs
    output [11:0] sel_out       // mux selection signal outputs
);
    
    reg [6:0] reg1;
    wire [11:0] sel_enc;
    reg [11:0] reg2;

    // first-level registers for retiming
    always @(posedge clk or negedge reset_n) begin
        if (!reset_n) reg1 <= 7'b0;
        else reg1 <= sel_in;
    end

    // encoder logic (must be placed after retiming)
    assign sel_enc[3:0] = reg1[4] ? ~reg1[3:0] : reg1[3:0];
    assign sel_enc[7:4] = {reg1[6],reg1[6],~reg1[6],~reg1[6]}&{reg1[5],~reg1[5],reg1[5],~reg1[5]};
    assign sel_enc[11:8] = reg1[4] ? {sel_enc[6:4],sel_enc[7]} : sel_enc[7:4];

    // second-level registers for retiming
    always @(posedge clk or negedge reset_n) begin
        if (!reset_n) reg2 <= 12'b0001_0001_0000;
        else reg2 <= sel_enc;
    end

    // final output
    assign sel_out = reg2;

endmodule

module ph_relaxation_osc #(
    `parameter_real(scale_factor, 2),           // frequency multiplication
    `parameter_real(f_range_cof_0 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=0
    `parameter_real(f_range_cof_1 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=1
    `parameter_real(f_range_cof_2 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=2
    `parameter_real(f_range_cof_3 [0:2], '{0.00,0.00,0.00})  // frequency tuning coefficients for f_range=3
)(
    input vdd,
    input vss,
    input [1:0] f_range,        // operation frequency range
    input [5:0] fdc_in,         // code converted from frequency
    input reset_n,              // reset (active low)
    output clk_relax_osc        // oscillation frequnecy
);

    real freq;
    xreal freq_xr;
    xbit clk_relax_osc_xbit;

    always @(*) begin
        if($realtime != 0) begin
            if(!reset_n) begin
                freq = 0;
            end
            else begin
                case(f_range)
                    2'b00 : freq = scale_factor * `pow(f_range_cof_0[0] + f_range_cof_0[1] * real'(fdc_in) + f_range_cof_0[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6;
                    2'b01 : freq = scale_factor * `pow(f_range_cof_1[0] + f_range_cof_1[1] * real'(fdc_in) + f_range_cof_1[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6;
                    2'b10 : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1] * real'(fdc_in) + f_range_cof_2[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6;
                    2'b11 : freq = scale_factor * `pow(f_range_cof_3[0] + f_range_cof_3[1] * real'(fdc_in) + f_range_cof_3[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6;
                    default : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1] * real'(fdc_in) + f_range_cof_2[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6;
                endcase
            end
        end
        else begin
            case(f_range)
                2'b00 : freq = scale_factor * `pow(f_range_cof_0[0] + f_range_cof_0[1]*63.0 + f_range_cof_0[2]*63.0*63.0, 0.5) * 1e6;
                2'b01 : freq = scale_factor * `pow(f_range_cof_1[0] + f_range_cof_1[1]*63.0 + f_range_cof_1[2]*63.0*63.0, 0.5) * 1e6;
                2'b10 : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5) * 1e6;
                2'b11 : freq = scale_factor * `pow(f_range_cof_3[0] + f_range_cof_3[1]*63.0 + f_range_cof_3[2]*63.0*63.0, 0.5) * 1e6;
            endcase
        end
    end

    real_to_xreal conn0(.in(freq), .out(freq_xr));
    freq_to_clk freq2clk(.in(freq_xr), .out(clk_relax_osc_xbit));
    xbit_to_bit conn1(.in(clk_relax_osc_xbit), .out(clk_relax_osc));

endmodule

module ph_rotator_core #(
    `parameter_real(f_range_cof_0 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=0
    `parameter_real(f_range_cof_1 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=1
    `parameter_real(f_range_cof_2 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=2
    `parameter_real(f_range_cof_3 [0:2], '{0.00,0.00,0.00})  // frequency tuning coefficients for f_range=3
)(
    input vdd,
    input vss,
    input reset_n,              // reset (active low)
    `input_xbit [7:0] clk_in,   // multiphase clock inputs
    input [6:0] phsel,          // phase select input
    input [1:0] f_range,        // frequency range
    input [5:0] f_bias,         // frequency calibration code
    input [5:0] fdc_ctrl,       // control code of relaxation osc for FDC
    `output_xbit clk_out,       // clock output
    `output_xbit clk_outb,      // clock output (negative)
    output reg clk_fdc,         // clock for FDC
    output reg pulse            // pulse signal for FDC
);

    reg clk_in_bit;
    wire pi_sync_clk, pi_sync_clkb; // internal clock for timing synchorinization (caution : synthesis)
    wire [3:0] weight;
    wire [3:0] sel_even, sel_odd;
    wire [11:0] sel_out;

    // encoder and retimers for selection signals (digital)
    ph_mux_enc ph_mux_enc (
        .vdd(vdd), .vss(vss),
        .clk(pi_sync_clk), .clkb(pi_sync_clkb), .reset_n(reset_n),
        .sel_in(phsel), .sel_out(sel_out));

    assign weight = sel_out[3:0];
    assign sel_odd = sel_out[7:4];
    assign sel_even = sel_out[11:8];

    // phase multiplexers and interpolator (analog)
    ph_mixer #(.f_range_cof_0(f_range_cof_0),
               .f_range_cof_1(f_range_cof_1),
               .f_range_cof_2(f_range_cof_2),
               .f_range_cof_3(f_range_cof_3))
        ph_mixer (.vdd(vdd), .vss(vss),
                  .clk_in(clk_in), .reset_n(reset_n),
                  .f_range(f_range), .f_bias(f_bias),
                  .sel_even(sel_even), .sel_odd(sel_odd), .weight(weight),
                  .piclk(clk_out), .piclkb(clk_outb));

    xbit_to_bit conn0(.in(clk_out), .out(pi_sync_clk));
    assign pi_sync_clkb = ~pi_sync_clk;

    // relaxation oscillator and pulse generator for the frequency to digital converter (frequency auto calibration for constant Ktdc)
    ph_relaxation_osc #(.scale_factor(2),
                        .f_range_cof_0(f_range_cof_0),
                        .f_range_cof_1(f_range_cof_1),
                        .f_range_cof_2(f_range_cof_2),
                        .f_range_cof_3(f_range_cof_3))
            ph_relaxation_osc(.vdd(vdd), .vss(vss),
                    .f_range(f_range), .fdc_in(fdc_ctrl),
                    .reset_n(reset_n), .clk_relax_osc(clk_osc));

    xbit_to_bit conn(.in(clk_in[0]), .out(clk_in_bit));

    ph_fdc_controller ph_fdc_controller (
                        .f_range(f_range), .clk_in(clk_in_bit),
                        .clk_osc(clk_osc), .reset_n(reset_n),
                        .clk_fdc(clk_fdc), .pulse(pulse));

endmodule
