/*----------------------------------------------------------------------
TESTBENCH tb_ph_rotator.sv

= Purpose =

A testbench to test the operation of the phase rotator including phase mixer (mux + interpolator), encoder, and retimer.

= Description =

The testbench sweeps the timing offset of the phase selection code, and calculates DNL(differential nonlinearity) and INL(integral nonlinearity). 

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module tb_ph_rotator();

    `get_timescale

    parameter real freq = 125e6;        // frequency of the input clocks
    parameter real t_lock = 20.0e-6;    // locking time

    xbit [7:0] clk_in;       // 8 uniformly-spaced multiphase input clocks
    reg [1:0] f_range;      // operation frequency range (global)
    reg [6:0] phsel;        // phase selection code (sweep)
    reg reset_n;            // reset (active low)
    xbit clk_out;           // clock output (pos)
    xbit clk_outb;          // clock output (neg)

    reg meas_flag;
    reg polarity;
 
    reg vdd = 1;
    reg vss = 0;

    real period = 1/freq;
    real phase_step = 2.8125; // 360 deg / 128 steps
    real phase[255:0];      // rising(0 ~ 127) + falling(128~225)

    real f_bias_r;
    real phsel_r;
    
    integer i;
    integer sel;

    // stimuli
    initial begin
        reset_n = 1'b1;
        f_range = 2'b01;
        #(1e-9/`TIME_SCALE) reset_n = 1'b0;
        #(2e-9/`TIME_SCALE) reset_n = 1'b1;
    end

    // multiphase clock generation
    clk_gen #(.freq(freq), .init_phase(2*M_PI*7/8)) clk_gen0(clk_in[0]);
    clk_gen #(.freq(freq), .init_phase(2*M_PI*6/8)) clk_gen1(clk_in[1]);
    clk_gen #(.freq(freq), .init_phase(2*M_PI*5/8)) clk_gen2(clk_in[2]);
    clk_gen #(.freq(freq), .init_phase(2*M_PI*4/8)) clk_gen3(clk_in[3]);
    clk_gen #(.freq(freq), .init_phase(2*M_PI*3/8)) clk_gen4(clk_in[4]);
    clk_gen #(.freq(freq), .init_phase(2*M_PI*2/8)) clk_gen5(clk_in[5]);
    clk_gen #(.freq(freq), .init_phase(2*M_PI*1/8)) clk_gen6(clk_in[6]);
    clk_gen #(.freq(freq), .init_phase(2*M_PI*0/8)) clk_gen7(clk_in[7]);

    // phsel generator (increasing and decreasing)
    initial begin
        phsel = 0;
        sel = 0;
        polarity = 0;

        #(t_lock/`TIME_SCALE);

        while (sel < 127) begin
            @(posedge `value(clk_out)); #(100.0e-12/`TIME_SCALE);
            polarity = 0;
            phsel = phsel + 1;
            sel = sel + 1;
        end
        while (sel > 0) begin
            @(posedge `value(clk_out)); #(100.0e-12/`TIME_SCALE);
            polarity = 1;
            phsel <= phsel - 1;
            sel = sel - 1;
        end

        $finish;
    end

    always @(phsel or DUT.f_bias) begin
        phsel_r = real'(phsel);
        f_bias_r = real'(DUT.f_bias);
    end

    // phase rotator (DUT)
        ph_rotator DUT(
            .vdd(vdd),
            .vss(vss),
            .f_range(f_range),
            .clk_in(clk_in),    // multiphase clock input
            .phsel(phsel),      // phase select input (fixed)
            .reset_n(reset_n),
            .clk_out(clk_out),  // clock output (reference phase)
            .clk_outb(clk_outb)
          );

    // probing
    probe_xbit prb_clk_in0(clk_in[0]);
    probe_xbit prb_clk_in1(clk_in[1]);
    probe_xbit prb_clk_in2(clk_in[2]);
    probe_xbit prb_clk_in3(clk_in[3]);
    probe_xbit prb_clk_in4(clk_in[4]);
    probe_xbit prb_clk_in5(clk_in[5]);
    probe_xbit prb_clk_in6(clk_in[6]);
    probe_xbit prb_clk_in7(clk_in[7]);
    probe_real prb_phsel(phsel_r);
    probe_xbit prb_clk_out(clk_out);
    probe_xbit prb_clk_outb(clk_outb);
    probe_phase #(.freq(freq), .phase_wrap(0)) prb_clk_out_phase(clk_out);
    probe_bit prb_reset_n(reset_n);

endmodule 
