/*----------------------------------------------------------------------
MODULE gro_tdc.sv

= Purpose =

A gated relaxation oscillator type time-to-digital converter.

= Description =

The gated relaxation oscillator type time-to-digital converter block.

= Revisions =

$Author$
$DateTIme$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module gro_tdc #(
    `parameter_real(K_tdc, 64),             // TDC gain
    `parameter_real(delay_rst, 131e-12),    // PFD reset propagation delay
    `parameter_real(delay_cq, 315e-12),     // clock-to-output delay
    `parameter_real(delay_cp, 202e-12),     // clock-to-proportional path output delay
    `parameter_real(f_range_cof_0 [0:2], '{141,400,6.38}),       // frequency tuning coefficients for f_range=0
    `parameter_real(f_range_cof_1 [0:2], '{-934,6780,29.2}),     // frequency tuning coefficients for f_range=1
    `parameter_real(f_range_cof_2 [0:2], '{148e3,110e3,-268}),   // frequency tuning coefficients for f_range=2
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 2.1e4),        // equivalent output resistance
    `parameter_real(res_out_prop, 0.26e4)   // equivalent output resistance (proportional term)
)(
    input vdd,
    input vss,
    input reset_n,          // reset (active low)
    input [1:0] f_range,    // frequency range
    `input_xbit clk_ref,    // reference clock input
    `input_xbit clk_fb,     // feedback clock input
    output reg [5:0] up,    // timing error output (up signal)
    output reg [5:0] dn,    // timing error output (dn signal)
    `output_xbit prop_up,   // PFD output pulse (up signal)
    `output_xbit prop_dn    // PFD output pulse (dn signal)
);

    // parameters
    parameter real prop_up_delay = delay_cp;
    parameter real prop_dn_delay = delay_cp;

    //variables
    wire pulse;
    reg clk_fdc;
    reg [5:0] f_bias, fdc_ctrl;

    xbit_to_bit conn0(.in(clk_ref), .out(clk_fdc));

    // GRO_TDC core
    gro_tdc_core #(.delay_rst(delay_rst),
                   .delay_cq(delay_cq),
                   .prop_up_delay(prop_up_delay),
                   .prop_dn_delay(prop_dn_delay),
                   .K_tdc(K_tdc),
                   .f_range_cof_0(f_range_cof_0),
                   .f_range_cof_1(f_range_cof_1),
                   .f_range_cof_2(f_range_cof_2))
            gro_tdc_core (
                .vdd(vdd), .vss(vss),
                .reset_n(reset_n), .f_range(f_range),
                .f_bias(f_bias), .fdc_ctrl(fdc_ctrl),
                .clk_ref(clk_ref), .clk_fb(clk_fb),
                .up(up), .dn(dn), .prop_up(prop_up), .prop_dn(prop_dn),
                .pulse(pulse));

    // FDC logic
    tdc_fdc_logic tdc_fdc_logic (
                    .clk(clk_fdc), .reset_n(reset_n), .pulse(pulse),
                    .dctrl(fdc_ctrl), .out(f_bias));

endmodule

module gray_counter (
    `input_xbit clk,
    input reset_n,
    output reg [4:0] gray_out
);
    reg [4:0] gray_in;

    always @(posedge `value(clk) or negedge reset_n) begin
        if(!reset_n) begin
            gray_in <= 5'b0;
        end
        else begin
            gray_in <= gray_in + 1;
        end
    end

    always @(gray_in) begin
        case(gray_in)
            5'b00000 : gray_out = 5'b00000;
            5'b00001 : gray_out = 5'b00001;
            5'b00010 : gray_out = 5'b00011;
            5'b00011 : gray_out = 5'b00010;
            5'b00100 : gray_out = 5'b00110;
            5'b00101 : gray_out = 5'b00111;
            5'b00110 : gray_out = 5'b00101;
            5'b00111 : gray_out = 5'b00100;
                                     
            5'b01000 : gray_out = 5'b01100;
            5'b01001 : gray_out = 5'b01101;
            5'b01010 : gray_out = 5'b01111;
            5'b01011 : gray_out = 5'b01110;
            5'b01100 : gray_out = 5'b01010;
            5'b01101 : gray_out = 5'b01011;
            5'b01110 : gray_out = 5'b01001;
            5'b01111 : gray_out = 5'b01000;
                                     
            5'b10000 : gray_out = 5'b11000;
            5'b10001 : gray_out = 5'b11001;
            5'b10010 : gray_out = 5'b11011;
            5'b10011 : gray_out = 5'b11010;
            5'b10100 : gray_out = 5'b11110;
            5'b10101 : gray_out = 5'b11111;
            5'b10110 : gray_out = 5'b11101;
            5'b10111 : gray_out = 5'b11100;
                                     
            5'b11000 : gray_out = 5'b10100;
            5'b11001 : gray_out = 5'b10101;
            5'b11010 : gray_out = 5'b10111;
            5'b11011 : gray_out = 5'b10110;
            5'b11100 : gray_out = 5'b10010;
            5'b11101 : gray_out = 5'b10011;
            5'b11110 : gray_out = 5'b10001;
            5'b11111 : gray_out = 5'b10000;
            default  : gray_out = 5'b00000;
        endcase
    end

endmodule

module gray_to_bin (
    input [4:0] in,
    output reg [4:0] out
);

    always @(in) begin
        case(in)
            5'b00000 : out = 5'b00000;
            5'b00001 : out = 5'b00001;
            5'b00011 : out = 5'b00010;
            5'b00010 : out = 5'b00011;
            5'b00110 : out = 5'b00100;
            5'b00111 : out = 5'b00101;
            5'b00101 : out = 5'b00110;
            5'b00100 : out = 5'b00111;

            5'b01100 : out = 5'b01000;
            5'b01101 : out = 5'b01001;
            5'b01111 : out = 5'b01010;
            5'b01110 : out = 5'b01011;
            5'b01010 : out = 5'b01100;
            5'b01011 : out = 5'b01101;
            5'b01001 : out = 5'b01110;
            5'b01000 : out = 5'b01111;

            5'b11000 : out = 5'b10000;
            5'b11001 : out = 5'b10001;
            5'b11011 : out = 5'b10010;
            5'b11010 : out = 5'b10011;
            5'b11110 : out = 5'b10100;
            5'b11111 : out = 5'b10101;
            5'b11101 : out = 5'b10110;
            5'b11100 : out = 5'b10111;

            5'b10100 : out = 5'b11000;
            5'b10101 : out = 5'b11001;
            5'b10111 : out = 5'b11010;
            5'b10110 : out = 5'b11011;
            5'b10010 : out = 5'b11100;
            5'b10011 : out = 5'b11101;
            5'b10001 : out = 5'b11110;
            5'b10000 : out = 5'b11111;
            default  : out = 5'b00000;
        endcase
    end

endmodule

module gro #(
    `parameter_real(K_tdc, 64),                 // TDC resolution
    `parameter_real(f_range_cof_0 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=0
    `parameter_real(f_range_cof_1 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=1
    `parameter_real(f_range_cof_2 [0:2], '{0.00,0.00,0.00})  // frequency tuning coefficients for f_range=2
)(
    input vdd,
    input vss,
    `input_xbit pw_in,          // input pulse width
    input bit [1:0] f_range,    // frequency range
    input bit [5:0] fdc_in,     // code converted from frequency
    input reset_n,              // reset (active low)
    `output_xbit outp,          // output clock whose period is proportional to the input pulse width (positive)
    `output_xbit outn           // output clock whose period is proportional to the input pulse width (negative)

);

    parameter real scale_factor_2 = 14;

    integer K_tdc_integer;

    real freq;
    real freq_ktdc;
    xreal freq_ktdc_xr;
    xreal freq_gro;
    integer count;

    initial begin
        K_tdc_integer = integer'(K_tdc);
        count = 0;
    end

    always @(*) begin
        if($realtime != 0) begin
            if(!reset_n) begin
                freq = 0;
            end
            else begin
                case(f_range)
                    2'b00 : freq = `pow(f_range_cof_0[0] + f_range_cof_0[1] * real'(fdc_in) + f_range_cof_0[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6 / scale_factor_2;
                    2'b01 : freq = `pow(f_range_cof_1[0] + f_range_cof_1[1] * real'(fdc_in) + f_range_cof_1[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6 / scale_factor_2;
                    2'b10 : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1] * real'(fdc_in) + f_range_cof_2[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6 / scale_factor_2;
                    default : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1] * real'(fdc_in) + f_range_cof_2[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6 / scale_factor_2;
                endcase
            end
        end
        else begin
            case(f_range)
                2'b00 : freq = `pow(f_range_cof_0[0] + f_range_cof_0[1]*63.0 + f_range_cof_0[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                2'b01 : freq = `pow(f_range_cof_1[0] + f_range_cof_1[1]*63.0 + f_range_cof_1[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                2'b10 : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                default : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;

            endcase
        end
    end

    always @(*) begin
        freq_ktdc = freq * K_tdc_integer / 2.0;
    end

    real_to_xreal conn0(.in(freq_ktdc), .out(freq_ktdc_xr));

    select select_freq(.sel(pw_in), .in({freq_ktdc_xr,`zero_xreal}), .out(freq_gro));

    freq_to_clk #(.num_phase(2), .init_phase(0.0))
            freq2clk(.in(freq_gro), .out({outp,outn}));

endmodule

module gro_tdc_core #(
    `parameter_real(K_tdc, 64),                 // TDC gain
    `parameter_real(delay_rst, 10.0e-12),       // PFD reset propagation delay
    `parameter_real(delay_cq, 100.0e-12),       // clock-to-output delay
    `parameter_real(prop_up_delay, 150.0e-12),  // clock-to-proportional path output (up) delay
    `parameter_real(prop_dn_delay, 150.0e-12),  // clock-to-proportional path output (dn) delay
    `parameter_real(f_range_cof_0 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=0
    `parameter_real(f_range_cof_1 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=1
    `parameter_real(f_range_cof_2 [0:2], '{0.00,0.00,0.00})  // frequency tuning coefficients for f_range=2
)(
    input vdd,
    input vss,
    input reset_n,          // reset (active low)
    input [1:0] f_range,    // frequency range
    input [5:0] f_bias,     // frequency calibration code
    input [5:0] fdc_ctrl,   // control code of relaxation osc for FDC
    `input_xbit clk_ref,    // reference clock input
    `input_xbit clk_fb,     // feedback clock input
    output reg [5:0] up,    // timing error output (up signal)
    output reg [5:0] dn,    // timing error output (dn signal)
    `output_xbit prop_up,   // PFD output pulse (up signal)
    `output_xbit prop_dn,   // PFD output pulse (dn signal)
    output reg pulse        // pulse signal to operate fdc_logic
);

    xbit up_pfd, dn_pfd;

    xbit clk_up_p, clk_up_n, clk_dn_p, clk_dn_n;

    reg [4:0] count_up0_gray, count_up1_gray, count_dn0_gray, count_dn1_gray;
    reg [4:0] up_sample0, up_sample1, dn_sample0, dn_sample1;
    reg [4:0] up_sample0_gray, up_sample1_gray, dn_sample0_gray, dn_sample1_gray;
    reg [4:0] up_prev0, up_prev1, dn_prev0, dn_prev1;
    reg [4:0] up0, up1, dn0, dn1;
    reg [5:0] up_sum, dn_sum;

    gro_tdc_pfd #(.delay_rst(delay_rst))
        pfd(.up(up_pfd), .dn(dn_pfd),
        .clk_ref(clk_ref), .clk_fb(clk_fb));

    gro #(.K_tdc(K_tdc),
          .f_range_cof_0(f_range_cof_0),
          .f_range_cof_1(f_range_cof_1),
          .f_range_cof_2(f_range_cof_2))
        gro_up(.vdd(vdd), .vss(vss),
            .pw_in(up_pfd), .f_range(f_range), .fdc_in(f_bias),
            .reset_n(reset_n), .outp(clk_up_p), .outn(clk_up_n));
    gro #(.K_tdc(K_tdc),
          .f_range_cof_0(f_range_cof_0),
          .f_range_cof_1(f_range_cof_1),
          .f_range_cof_2(f_range_cof_2))
        gro_dn(.vdd(vdd), .vss(vss),
            .pw_in(dn_pfd), .f_range(f_range), .fdc_in(f_bias),
            .reset_n(reset_n), .outp(clk_dn_p), .outn(clk_dn_n));
    
    gray_counter gray_up0(.clk(clk_up_p), .reset_n(reset_n), .gray_out(count_up0_gray));
    gray_counter gray_up1(.clk(clk_up_n), .reset_n(reset_n), .gray_out(count_up1_gray));
    gray_counter gray_dn0(.clk(clk_dn_p), .reset_n(reset_n), .gray_out(count_dn0_gray));
    gray_counter gray_dn1(.clk(clk_dn_n), .reset_n(reset_n), .gray_out(count_dn1_gray));

    gray_to_bin bin_up0(.in(up_sample0_gray), .out(up_sample0));
    gray_to_bin bin_up1(.in(up_sample1_gray), .out(up_sample1));
    gray_to_bin bin_dn0(.in(dn_sample0_gray), .out(dn_sample0));
    gray_to_bin bin_dn1(.in(dn_sample1_gray), .out(dn_sample1));

    assign up0 = up_sample0 + ~up_prev0 + 5'b00001;
    assign up1 = up_sample1 + ~up_prev1 + 5'b00001;
    assign dn0 = dn_sample0 + ~dn_prev0 + 5'b00001;
    assign dn1 = dn_sample1 + ~dn_prev1 + 5'b00001;
    assign up_sum = {1'b0,up0} + {1'b0,up1};
    assign dn_sum = {1'b0,dn0} + {1'b0,dn1};

    // counted value sampling (timing error between two adjacent negative edges of the feedback clock)
    always @(negedge `value(clk_fb) or negedge reset_n) begin
        if(!reset_n) begin
            up_prev0 <= 5'b0;
            up_prev1 <= 5'b0;
            up_sample0_gray <= 5'b0;
            up_sample1_gray <= 5'b0;
        end
        else begin
            up_prev0 <= up_sample0;
            up_prev1 <= up_sample1;
            up_sample0_gray <= count_up0_gray;
            up_sample1_gray <= count_up1_gray;
        end
    end

    always @(negedge `value(clk_fb) or negedge reset_n) begin
        if(!reset_n) begin
            dn_prev0 <= 5'b0;
            dn_prev1 <= 5'b0;
            dn_sample0_gray <= 5'b0;
            dn_sample1_gray <= 5'b0;
        end
        else begin
            dn_prev0 <= dn_sample0;
            dn_prev1 <= dn_sample1;
            dn_sample0_gray <= count_dn0_gray;
            dn_sample1_gray <= count_dn1_gray;
        end
    end

    // output retiming
    always @(posedge `value(clk_fb) or negedge reset_n) begin
        if(!reset_n) begin
            up <= 6'b0;
            dn <= 6'b0;
        end
        else begin
            up <= #(delay_cq/`TIME_SCALE) up_sum;
            dn <= #(delay_cq/`TIME_SCALE) dn_sum;
        end
    end

    // PFD output pulse
    delay_xbit #(.delay(prop_up_delay)) up_delay(.in(up_pfd), .out(prop_up));
    delay_xbit #(.delay(prop_dn_delay)) dn_delay(.in(dn_pfd), .out(prop_dn));

    // relaxation oscillator and pulse generator for the frequency to digital converter (frequency auto calibration for constant Ktdc)
    tdc_relaxation_osc #(.scale_factor(K_tdc/2),
                         .f_range_cof_0(f_range_cof_0),
                         .f_range_cof_1(f_range_cof_1),
                         .f_range_cof_2(f_range_cof_2))
            tdc_relaxation_osc(.vdd(vdd), .vss(vss),
                    .f_range(f_range), .fdc_in(fdc_ctrl),
                    .reset_n(reset_n), .clk_relax_osc(clk_osc));

    xbit_to_bit conn(.in(clk_ref), .out(clk_fdc));

    tdc_fdc_controller #(.K_tdc(K_tdc))
            tdc_fdc_controller(.vdd(vdd), .vss(vss), .clk_in(clk_fdc),
                    .clk_osc(clk_osc), .reset_n(reset_n), .pulse(pulse));

endmodule

module gro_tdc_pfd #(
    `parameter_real(delay_rst, 0.0)     // reset propagation delay
)(
    `output_xbit up,	                // up signal
    `output_xbit dn,	                // down signal
    `input_xbit clk_ref,                // reference clock
    `input_xbit clk_fb                  // feedback clock
);

    // variables
    xbit rst;

    // two dffs with asynchronous reset
    dff_rst_async_xbit #(.init_value(0)) dff_up(.q(up), .d(`one_xbit), .clk(clk_ref), .rst(rst));
    dff_rst_async_xbit #(.init_value(0)) dff_dn(.q(dn), .d(`one_xbit), .clk(clk_fb), .rst(rst));

    // reset path
    and2_xbit #(.delay(delay_rst)) and_xbit(.out(rst),.in_a(up),.in_b(dn));

endmodule

module tdc_fdc_acc (
    input clk,
    input reset_n,
    input rst_sync_global,
    input rst_sync_local,
    input enable,
    input [4:0] in,
    output reg [3:0] n_up,
    output reg [3:0] n_dn
);

    reg [3:0] n_up_nxt, n_dn_nxt;
    reg [1:0] flag;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) flag <= 2'b0;
        else begin
            if(rst_sync_local) begin
                flag <= 2'b0;
            end
            else begin
                if(in == 5'b01110) begin     // 15 - 1
                    if(flag[1]) begin
                        flag <= 2'b11;
                    end
                    else begin
                        flag <= 2'b01;
                    end
                end
                else begin
                    if(in == 5'b01111) begin // 15+1 - 1
                        flag <= 2'b11;
                    end
                    else begin
                        flag <= flag;
                    end
                end
            end
        end
    end
    
    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            n_up <= 4'b0;
            n_dn <= 4'b0;
        end
        else begin
            if(rst_sync_global) begin
                n_up <= 4'b0;
                n_dn <= 4'b0;
            end
            else begin
                if(enable) begin
                    n_up <= n_up_nxt;
                    n_dn <= n_dn_nxt;
                end
                else begin
                    n_up <= n_up;
                    n_dn <= n_dn;
                end
            end
        end
    end

    always @(*) begin
        case(flag)
            2'b00 :
                begin
                    n_up_nxt = n_up;
                    n_dn_nxt = n_dn + 1;
                end
            2'b11 :
                begin
                    n_up_nxt = n_up + 1;
                    n_dn_nxt = n_dn;
                end
            default :
                begin
                    n_up_nxt = n_up;
                    n_dn_nxt = n_dn;
                end
        endcase
    end

endmodule

module tdc_fdc_controller #(
    `parameter_real(K_tdc, 64)
)(
    input vdd,
    input vss,
    input clk_in,               // clock whose frequency is going to be acquired
    input clk_osc,              // oscillator operating frequency
    input reset_n,              // reset (active low)
    output reg pulse            // pulse signal to operate fdc_logic
);

    reg clk_osc_div_ktdc, clk_osc_div;

    tdc_fdc_div #(.div_factor(K_tdc/2))
            fdc_div_ktdc(.clk(clk_osc), .reset_n(reset_n),
                         .clk_div(clk_osc_div_ktdc));

    tdc_fdc_div #(.div_factor(16))
            fdc_div(.clk(clk_osc_div_ktdc), .reset_n(reset_n),
                    .clk_div(clk_osc_div));

    tdc_fdc_pulse_gen fdc_pulse_gen(.clk(clk_in), .in(clk_osc_div), .reset_n(reset_n), .pulse(pulse));


endmodule

module tdc_fdc_counter (
    input clk,
    input reset_n,
    input rst_sync,
    output reg [4:0] cnt
);

    reg [4:0] cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 5'b0;
        end
        else begin
            if(rst_sync) begin
                cnt <= 5'b0;
            end
            else begin
                cnt <= cnt_nxt;
            end
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module tdc_fdc_counter_en (
    input clk,
    input reset_n,
    input enable,
    output reg [3:0] cnt
);

    reg [3:0] cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 4'b0;
        end
        else begin
            if(enable)
                if(cnt == 4'b1110) cnt <= 4'b0;
                else cnt <= cnt_nxt;
            else
                cnt <= cnt;
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module tdc_fdc_decision (
    input clk,
    input reset_n,
    input enable,
    input enable_global,
    input [3:0] n_up,
    input [3:0] n_dn,
    output reg [5:0] interior_code,
    output reg [5:0] out
);

    wire change;
    reg n_up_msb_pre, n_dn_msb_pre;
    reg [5:0] interior_code_nxt, exterior_code_nxt, history;

    assign change = ((n_up[3]^n_up_msb_pre) | (n_dn[3]^n_dn_msb_pre)) ? 1'b1 : 1'b0;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            interior_code <= 6'b100000; //6'b111111;
            out <= 6'b100000; //6'b111111;
        end
        else begin
            out <= exterior_code_nxt;
            if(enable) interior_code <= interior_code_nxt;
            else interior_code <= interior_code;
        end
    end

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            n_up_msb_pre <= 1'b0;
            n_dn_msb_pre <= 1'b0;
        end
        else begin
            if(enable_global) begin
                n_up_msb_pre <= n_up[3];
                n_dn_msb_pre <= n_dn[3];
            end
            else begin
                n_up_msb_pre <= n_up_msb_pre;
                n_dn_msb_pre <= n_dn_msb_pre;
            end
        end
    end

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            history <= 6'b0;
        end
        else begin
            if(enable_global) begin
                if(change) history <= {history[5:0],1'b1};
                else history <= {history[5:0],1'b0};
            end
            else begin
                history <= history;
            end
        end
    end

    always @(*) begin
        if(n_up[3]) begin
            if(interior_code == 6'b111111) interior_code_nxt = 6'b111111;
            else interior_code_nxt = interior_code + 1;
        end
        else if(n_dn[3]) begin
            if(interior_code == 6'b000000) interior_code_nxt = 6'b000000;
            else interior_code_nxt = interior_code - 1;
        end
        else begin
            interior_code_nxt = interior_code;
        end
    end

    always @(*) begin
        if(&history) exterior_code_nxt = out;
        else exterior_code_nxt = interior_code;
    end

endmodule

module tdc_fdc_div #(
    `parameter_integer(div_factor, 32)
)(
    input clk,
    input reset_n,
    output reg clk_div
);

    reg [9:0] cnt, cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 0;
            clk_div <= 0;
        end
        else begin
            if(cnt == div_factor/2-1) begin
                cnt <= 0;
                clk_div <= ~clk_div;
            end
            else begin
                cnt <= cnt_nxt;
            end
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module tdc_fdc_logic (
    input clk,                  // clock whose frequency is going to be acquired
    input reset_n,              // reset (active low)
    input pulse,                // pulse detecting the rising edge of the divided clk_osc
    output reg [5:0] dctrl,     // control code only in FDC
    output reg [5:0] out        // control code output
);

    wire en_cnt, en_acc, en_dec, en_dec_global, rst_cnt, rst_acc_global, rst_acc_local;
    reg period, period_retimed, pulse_acc, pulse_retimed;
    reg [4:0] cnt;
    reg [3:0] n_up, n_dn, cnt_period;

    assign en_cnt = pulse;
    assign en_acc = pulse;
    assign en_dec = period;
    assign en_dec_global = pulse_acc;
    assign rst_cnt = pulse;
    assign rst_acc_global = pulse_acc;
    assign rst_acc_local = pulse_retimed;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            pulse_retimed <= 1'b0;
            period_retimed <= 1'b0;
        end
        else begin
            pulse_retimed <= pulse;
            period_retimed <= period;
        end
    end

    always @(*) begin
        pulse_acc = period&(~period_retimed);
        if(cnt_period == 4'b1110) period = 1'b1;
        else period = 1'b0;
    end 

    tdc_fdc_counter fdc_counter_fd(.clk(clk), .reset_n(reset_n), .rst_sync(rst_cnt), .cnt(cnt));

    tdc_fdc_counter_en fdc_counter_acc(.clk(clk), .reset_n(reset_n), .enable(en_cnt), .cnt(cnt_period));

    tdc_fdc_acc fdc_acc(.clk(clk), .reset_n(reset_n), .rst_sync_global(rst_acc_global), .rst_sync_local(rst_acc_local), .enable(en_acc), .in(cnt), .n_up(n_up), .n_dn(n_dn));

    tdc_fdc_decision fdc_decision(.clk(clk), .reset_n(reset_n), .enable(en_dec), .enable_global(en_dec_global), .n_up(n_up), .n_dn(n_dn), .interior_code(dctrl), .out(out));

endmodule

module tdc_fdc_pulse_gen (
    input clk,
    input in,
    input reset_n,
    output reg pulse
);

    reg reg1, reg2;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            reg1 <= 1'b0;
            reg2 <= 1'b0;
        end
        else begin
            reg1 <= in;
            reg2 <= reg1;
        end
    end

    always @(*) begin
        pulse = reg1&(~reg2);
    end

endmodule

module tdc_relaxation_osc #(
    `parameter_real(scale_factor, 32),          // frequency multiplication
    `parameter_real(f_range_cof_0 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=0
    `parameter_real(f_range_cof_1 [0:2], '{0.00,0.00,0.00}), // frequency tuning coefficients for f_range=1
    `parameter_real(f_range_cof_2 [0:2], '{0.00,0.00,0.00})  // frequency tuning coefficients for f_range=2
)(
    input vdd,
    input vss,
    input [1:0] f_range,        // operation frequency range 
    input [5:0] fdc_in,         // code converted from frequency
    input reset_n,              // reset (active low)
    output clk_relax_osc        // oscillation frequnecy
);

    parameter real scale_factor_2 = 14;

    real freq;
    xreal freq_xr;
    xbit clk_relax_osc_xbit;

    always @(*) begin
        if($realtime != 0) begin
            if(!reset_n) begin
                freq = 0;
            end
            else begin
                case(f_range)
                    2'b00 : freq = scale_factor * `pow(f_range_cof_0[0] + f_range_cof_0[1] * real'(fdc_in) + f_range_cof_0[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6 / scale_factor_2;
                    2'b01 : freq = scale_factor * `pow(f_range_cof_1[0] + f_range_cof_1[1] * real'(fdc_in) + f_range_cof_1[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6 / scale_factor_2;
                    2'b10 : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1] * real'(fdc_in) + f_range_cof_2[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6 / scale_factor_2;
                    default : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1] * real'(fdc_in) + f_range_cof_2[2] * real'(fdc_in) * real'(fdc_in), 0.5) * 1e6 / scale_factor_2;
                endcase
            end
        end
        else begin
            case(f_range)
                2'b00 : freq = scale_factor * `pow(f_range_cof_0[0] + f_range_cof_0[1]*63.0 + f_range_cof_0[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                2'b01 : freq = scale_factor * `pow(f_range_cof_1[0] + f_range_cof_1[1]*63.0 + f_range_cof_1[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                2'b10 : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                default : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
            endcase
        end
    end

    real_to_xreal conn0(.in(freq), .out(freq_xr));
    freq_to_clk freq2clk(.in(freq_xr), .out(clk_relax_osc_xbit));
    xbit_to_bit conn1(.in(clk_relax_osc_xbit), .out(clk_relax_osc));

endmodule
