/*----------------------------------------------------------------------
MODEL divider.sv

= Purpose =

A clock frequency divider with a parameterized division factor.

= Description =

This divider is triggered by both the rising and falling edges of the input clock in order to keep the output duty cycle close to 50% even for odd division factors. For the dividing factor, 2 to 9 are available.

A fixed-timestep model.

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module divider #(
    `parameter_real(delay_cq, 275e-12),     // clock-to-output delay
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 2.1e4)         // equivalent output resistance
)(
    input vdd,
    input vss,
    input reset_n,          // reset signal (active low)
    input [2:0] div_sel,    // dividing factor selection code (2,3,4,5,6,7,8,9)
    `input_xbit clk_inp,    // input positive clock
    `input_xbit clk_inn,    // input negative clock
    `output_xbit clk_outp,  // output positive clock 
    `output_xbit clk_outn   // output negative clock 
);

    wire [3:0] next;
    wire enable;
    reg [2:0] div_sel_retimed;
    reg [3:0] div_factor;
    reg [3:0] count;

    xbit rst, en;
    xbit d0;
//    xbit out0, out1;
    
    always @(posedge `value(clk_outp) or negedge reset_n) begin
        if(!reset_n) begin
            div_sel_retimed <= 3'b000;
        end    
        else begin
            div_sel_retimed <= div_sel;
        end
    end

    always @(*) begin
        case(div_sel_retimed)
            3'b000 : div_factor = 1;    // dividing factor = 2
            3'b001 : div_factor = 2;    // dividing factor = 3
            3'b010 : div_factor = 3;    // dividing factor = 4
            3'b011 : div_factor = 4;    // dividing factor = 5
            3'b100 : div_factor = 5;    // dividing factor = 6
            3'b101 : div_factor = 6;    // dividing factor = 7
            3'b110 : div_factor = 7;    // dividing factor = 8
            3'b111 : div_factor = 8;    // dividing factor = 9
            default : div_factor = 1;   // dividing factor = 2
        endcase
    end

    assign next = (count == div_factor) ? 0 : (count + 1);
    assign enable = (count == div_factor) ? 1'b1 : 1'b0;

    always @(posedge `value(clk_inp) or posedge `value(clk_inn) or negedge reset_n) begin
        if (!reset_n) begin
            count <= 4'b1111;
        end
        else begin
            #(delay_cq/`TIME_SCALE) count <= next;
        end
    end

    bit_to_xbit conn0(.in(enable), .out(en));
    bit_to_xbit conn1(.in(~reset_n), .out(rst));

    div_dff_dual_edge #(.delay_cq(delay_cq))
            dff_dual_edge(.d(d0), .rst(rst),
                .clkp(clk_inp), .clkn(clk_inn), .q(clk_outn));

    inv_xbit inv0(.in(clk_outn), .out(clk_outp));

    mux_xbit mux0(.in({clk_outp,clk_outn}), .sel(en), .out(d0));

endmodule

module div_dff_dual_edge #(
    `parameter_real(delay_cq, 100e-12)  // clk-to-q delay
)(
    `input_xbit d,
    `input_xbit rst,
    `input_xbit clkp,
    `input_xbit clkn,
    `output_xbit q
);

    xbit q0, q1;
    xbit clkn_b, clk_or;

    dff_rst_async_xbit  #(.delay_cq(0.0))
                dff0(.d(d), .clk(clkp), .rst(rst), .q(q0));
    dff_rst_async_xbit  #(.delay_cq(0.0))
                dff1(.d(d), .clk(clkn), .rst(rst), .q(q1));

    inv_xbit    inv0(.in(clkn), .out(clkn_b));

    or_xbit    or0(.in({clkp,clkn_b}), .out(clk_or));

    mux_xbit #(.delay(delay_cq))
                mux0(.in({q0,q1}), .sel(clk_or), .out(q));

endmodule
