/*----------------------------------------------------------------------
MODEL bbpd.sv

= Purpose =

A bang-bang phase detector model to compare the phase difference between the two input clocks, providing the polarity information only.

= Description =

The bang-bang phase-frequency detector measures the phase differences between the two input clocks and expresses its polarity in binary outputs (up and down).

= Revisions =

$Author$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module bbpd #(
    `parameter_real(t_offset, 20e-12),          // zero-output timing offset
    `parameter_real(clk_to_q_delay, 189e-12),   // clock-to-output delay
    `parameter_real(cap_in, 1.8e-15),           // equivalent input capacitance
    `parameter_real(res_out, 2.1e4)             // equivalent output resistance
)(
    input vdd,
    input vss,
    `input_xbit clk_ref,    // reference clock input
    `input_xbit clk_fb,     // feedback clock input
    output up,              // output (up)
    output dn               // output (down)
);

    parameter real delay_ref = (t_offset > 0) ? t_offset : 0;
    parameter real delay_fb = (t_offset < 0) ? -t_offset : 0;

    // variables
    xbit clk_ref_delay, clk_fb_delay;
    xbit up_xbit, dn_xbit;

    // connectors
    xbit_to_bit #(.width(2)) conn(.in({up_xbit,dn_xbit}), .out({up,dn}));

    // d-flip flop
    delay_xbit #(.delay(delay_ref)) delay_clk_ref(.in(clk_ref), .out(clk_ref_delay));
    delay_xbit #(.delay(delay_fb)) delay_clk_fb(.in(clk_fb), .out(clk_fb_delay));

    dff_xbit #(.delay_cq(clk_to_q_delay)) dff(.d(clk_ref_delay), .clk(clk_fb_delay), .q(up_xbit));

    inv_xbit inv(.in(up_xbit), .out(dn_xbit));

endmodule
