`define input_bit       input bit
`define output_bit      output bit
`define input_real      input real
`define output_real     output real
`define input_xbit      input xbit
`define output_xbit     output xbit
`define inout_xbit      inout xbit
`define input_xreal     input xreal
`define output_xreal    output xreal
`define parameter_bit(__name,__value)       parameter bit __name = __value
`define parameter_integer(__name,__value)   parameter integer __name = __value
`define parameter_real(__name,__value)      parameter real __name = __value
`define parameter_string(__name,__value)    parameter string __name = __value
`define parameter_realarray(__name)         parameter real __name = 0
`define xbit(__name)    xbit __name
`define xreal(__name)    xreal __name
`define ground          1'b0
`define TIME_SCALE       1
`define one_xbit        1'b1
`define zero_xreal      1'b0
`define pow(val1,val2)  val1+val2
`define get_timescale
//systemVerilog HDL for "violin", "pad_vddt" "systemVerilog"

`include "xmodel.h"

module padv_vss (
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "pad_vddt" "systemVerilog"

`include "xmodel.h"

module padv_vddt (
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "pad_vddt" "systemVerilog"

`include "xmodel.h"

module padv_vdda_1 (
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "pad_vddt" "systemVerilog"

`include "xmodel.h"

module padv_vdda_0 (
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "padv_out_esd" "systemVerilog"

`include "xmodel.h"

module padv_out_esd (
    inout pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "padh_out_cmos_en" "systemVerilog"

`include "xmodel.h"

module padv_out_cmos_en (
    inout pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss,
    input out_en,
    input core_out
);

    assign pad = (out_en) ? core_out : 1'b0;

endmodule
//systemVerilog HDL for "violin", "padh_inout_cmos" "systemVerilog"

`include "xmodel.h"

module padv_inout_cmos (
    inout pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss,
    input out_en,
    input in_en,
    input core_out,
    output core_in
);

    assign pad = (out_en & !in_en) ? core_out : 1'bz ;
    assign core_in = (!out_en & in_en) ? pad : 1'b0;

endmodule
//systemVerilog HDL for "violin", "padv_in_esd" "systemVerilog"

`include "xmodel.h"

module padv_in_esd (
    inout pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss,
    output in
);

    assign in = pad;    

endmodule
//systemVerilog HDL for "violin", "padh_in_cmos" "systemVerilog"

`include "xmodel.h"

module padv_in_cmos (
    inout pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss,
    output core_in
);

    assign core_in = pad;

endmodule
//systemVerilog HDL for "violin", "padv_empty" "systemVerilog"

`include "xmodel.h"

module padv_empty (
    `inout_xbit pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "pad_vddt" "systemVerilog"

`include "xmodel.h"

module padh_vss (
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "pad_vddt" "systemVerilog"

`include "xmodel.h"

module padh_vddt (
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "pad_vddt" "systemVerilog"

`include "xmodel.h"

module padh_vdda_1 (
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);
endmodule
//systemVerilog HDL for "violin", "pad_vddt" "systemVerilog"

`include "xmodel.h"

module padh_vdda_0 (
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "padh_out_esd" "systemVerilog"

`include "xmodel.h"

module padh_out_esd (
    inout pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "padh_out_cmos_en" "systemVerilog"

`include "xmodel.h"

module padh_out_cmos_en (
    inout pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss,
    input out_en,
    input core_out
);

    assign pad = (out_en) ? core_out : 1'b0;

endmodule
//systemVerilog HDL for "violin", "padh_inout_cmos" "systemVerilog"

`include "xmodel.h"

module padh_inout_cmos (
    inout pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss,
    input out_en,
    input in_en,
    input core_out,
    output core_in
);

    assign pad = (out_en & !in_en) ? core_out : 1'bz ;
    assign core_in = (!out_en & in_en) ? pad : 1'b0;

endmodule
//systemVerilog HDL for "violin", "padh_in_esd" "systemVerilog"

`include "xmodel.h"

module padh_in_esd (
    inout pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss,
    output in
);

    assign in = pad;

endmodule
//systemVerilog HDL for "violin", "padh_in_cmos" "systemVerilog"

`include "xmodel.h"

module padh_in_cmos (
    inout pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss,
    output core_in
);

    assign core_in = pad;

endmodule
//systemVerilog HDL for "violin", "padh_empty" "systemVerilog"

`include "xmodel.h"

module padh_empty (
    `inout_xbit pad,
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "out_cml" "systemVerilog"

`include "xmodel.h"

module out_cml (
    input vddt,
    input vss,
    input [3:0] cmlbias_ctrlb,
    input bias_rstb,
    input cml_enb,
    `input_xbit inp,
    `input_xbit inn,
    `output_xbit outp,
    `output_xbit outn
);

    xbit cml_en;

    bit_to_xbit conn(.in(~cml_enb), .out(cml_en));

    and2_xbit and0(.in_a(cml_en), .in_b(inp), .out(outp));
    and2_xbit and1(.in_a(cml_en), .in_b(inn), .out(outn));

endmodule
//systemVerilog HDL for "violin", "pad_vddt" "systemVerilog"

`include "xmodel.h"

module pad_corner (
    input vdda_0,
    input vddt,
    input vdda_1,
    input vss
);

endmodule
//systemVerilog HDL for "violin", "in_csda" "systemVerilog"

`include "xmodel.h"

module in_csda (
    input vddt,
    input vss,
    `input_xbit inp,
    `input_xbit inn,
    `output_xbit out
);

    buf_xbit buf0(.in(inp), .out(out));

endmodule
/*----------------------------------------------------------------------
MODEL ph_rotator.sv

= Purpose =

A phase rotator block.

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

module ph_rotator #(
    `parameter_realarray(f_range_cof_0),  // frequency tuning coefficients for f_range=0
    `parameter_realarray(f_range_cof_1),       // frequency tuning coefficients for f_range=1
    `parameter_realarray(f_range_cof_2),     // frequency tuning coefficients for f_range=2
    `parameter_realarray(f_range_cof_3),   // frequency tuning coefficients for f_range=3
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 0.26e3)        // equivalent output resistance
)(
    input vdd,
    input vss,
    input reset_n,              // reset (active low)
    input [1:0] f_range,        // operating frequency range (global)
    `input_xbit [7:0] clk_in,   // multiphase clock inputs
    input [6:0] phsel,          // phase select input
    `output_xbit clk_out,       // clock output
    `output_xbit clk_outb       // clock output (negative)
);

    wire pulse;
    reg clk_fdc;
    reg [5:0] f_bias, fdc_ctrl;

    // PH_ROTATOR core
    ph_rotator_core #(.f_range_cof_0(f_range_cof_0),
                      .f_range_cof_1(f_range_cof_1),
                      .f_range_cof_2(f_range_cof_2),
                      .f_range_cof_3(f_range_cof_3))
        ph_rotator_core (
            .vdd(vdd), .vss(vss),
            .reset_n(reset_n),
            .f_range(f_range), .f_bias(f_bias), .fdc_ctrl(fdc_ctrl),
            .clk_in(clk_in), .phsel(phsel),
            .clk_out(clk_out), .clk_outb(clk_outb),
            .clk_fdc(clk_fdc), .pulse(pulse)
        );

    // FDC logic
    ph_fdc_logic ph_fdc_logic (
        .clk(clk_fdc), .reset_n(reset_n), .pulse(pulse),
        .dctrl(fdc_ctrl), .out(f_bias));

endmodule

module ph_fdc_acc (
    input clk,
    input reset_n,
    input rst_sync_global,
    input rst_sync_local,
    input enable,
    input [4:0] in,
    output reg [3:0] n_up,
    output reg [3:0] n_dn
);

    reg [3:0] n_up_nxt, n_dn_nxt;
    reg [1:0] flag;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) flag <= 2'b0;
        else begin
            if(rst_sync_local) begin
                flag <= 2'b0;
            end
            else begin
                if(in == 5'b01110) begin     // 15 - 1
                    if(flag[1]) begin
                        flag <= 2'b11;
                    end
                    else begin
                        flag <= 2'b01;
                    end
                end
                else begin
                    if(in == 5'b01111) begin // 15+1 - 1
                        flag <= 2'b11;
                    end
                    else begin
                        flag <= flag;
                    end
                end
            end
        end
    end
    
    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            n_up <= 4'b0;
            n_dn <= 4'b0;
        end
        else begin
            if(rst_sync_global) begin
                n_up <= 4'b0;
                n_dn <= 4'b0;
            end
            else begin
                if(enable) begin
                    n_up <= n_up_nxt;
                    n_dn <= n_dn_nxt;
                end
                else begin
                    n_up <= n_up;
                    n_dn <= n_dn;
                end
            end
        end
    end

    always @(*) begin
        case(flag)
            2'b00 :
                begin
                    n_up_nxt = n_up;
                    n_dn_nxt = n_dn + 1;
                end
            2'b11 :
                begin
                    n_up_nxt = n_up + 1;
                    n_dn_nxt = n_dn;
                end
            default :
                begin
                    n_up_nxt = n_up;
                    n_dn_nxt = n_dn;
                end
        endcase
    end

endmodule

module ph_fdc_controller (
    input [1:0] f_range,        // operation frequency range
    input clk_in,               // clock whose frequency is going to be acquired
    input clk_osc,              // oscillator operating frequency
    input reset_n,              // reset (active low)
    output reg clk_fdc,         // clock for fdc_logic
    output reg pulse            // pulse signal to operate fdc_logic
);

    reg clk_osc_div, clk_in1, clk_osc1, clk_osc_div2;
    wire clk_in_mux, clk_osc_mux;

    assign clk_in_mux = (f_range == 2'b11) ? clk_in1 : clk_in;
    assign clk_osc_mux = (f_range == 2'b11) ? clk_osc1 : clk_osc_div2;

    always @(clk_in_mux) begin
        clk_fdc = clk_in_mux;
    end

    ph_fdc_div #(.div_factor(2))
            fdc_div2_clk_osc(.clk(clk_osc), .reset_n(reset_n),
                             .clk_div(clk_osc_div2));

    ph_fdc_div #(.div_factor(8))
            fdc_div8_clk_in(.clk(clk_in), .reset_n(reset_n),
                            .clk_div(clk_in1));
    ph_fdc_div #(.div_factor(8))
            fdc_div8_clk_osc(.clk(clk_osc_div2), .reset_n(reset_n),
                             .clk_div(clk_osc1));

    ph_fdc_div #(.div_factor(16))
            fdc_div(.clk(clk_osc_mux), .reset_n(reset_n), .clk_div(clk_osc_div));

    ph_fdc_pulse_gen fdc_pulse_gen(.clk(clk_fdc), .in(clk_osc_div), .reset_n(reset_n), .pulse(pulse));

endmodule

module ph_fdc_counter (
    input clk,
    input reset_n,
    input rst_sync,
    output reg [4:0] cnt
);

    reg [4:0] cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 5'b0;
        end
        else begin
            if(rst_sync) begin
                cnt <= 5'b0;
            end
            else begin
                cnt <= cnt_nxt;
            end
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module ph_fdc_counter_en (
    input clk,
    input reset_n,
    input enable,
    output reg [3:0] cnt
);

    reg [3:0] cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 4'b0;
        end
        else begin
            if(enable)
                if(cnt == 4'b1110) cnt <= 4'b0;
                else cnt <= cnt_nxt;
            else
                cnt <= cnt;
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module ph_fdc_decision (
    input clk,
    input reset_n,
    input enable,
    input enable_global,
    input [3:0] n_up,
    input [3:0] n_dn,
    output reg [5:0] interior_code,
    output reg [5:0] out
);

    wire change;
    reg n_up_msb_pre, n_dn_msb_pre;
    reg [5:0] interior_code_nxt, exterior_code_nxt, history;

    assign change = ((n_up[3]^n_up_msb_pre) | (n_dn[3]^n_dn_msb_pre)) ? 1'b1 : 1'b0;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            interior_code <= 6'b100100;
            out <= 6'b100100;
        end
        else begin
            out <= exterior_code_nxt;
            if(enable) interior_code <= interior_code_nxt;
            else interior_code <= interior_code;
        end
    end

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            n_up_msb_pre <= 1'b0;
            n_dn_msb_pre <= 1'b0;
        end
        else begin
            if(enable_global) begin
                n_up_msb_pre <= n_up[3];
                n_dn_msb_pre <= n_dn[3];
            end
            else begin
                n_up_msb_pre <= n_up_msb_pre;
                n_dn_msb_pre <= n_dn_msb_pre;
            end
        end
    end

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            history <= 6'b0;
        end
        else begin
            if(enable_global) begin
                if(change) history <= {history[5:0],1'b1};
                else history <= {history[5:0],1'b0};
            end
            else begin
                history <= history;
            end
        end
    end

    always @(*) begin
        if(n_up[3]) begin
            if(interior_code == 6'b111111) interior_code_nxt = 6'b111111;
            else interior_code_nxt = interior_code + 1;
        end
        else if(n_dn[3]) begin
            if(interior_code == 6'b000000) interior_code_nxt = 6'b000000;
            else interior_code_nxt = interior_code - 1;
        end
        else begin
            interior_code_nxt = interior_code;
        end
    end

    always @(*) begin
        if(&history) exterior_code_nxt = out;
        else exterior_code_nxt = interior_code;
    end

endmodule

module ph_fdc_div #(
    `parameter_integer(div_factor, 8)
)(
    input clk,
    input reset_n,
    output reg clk_div
);

    reg [9:0] cnt, cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 0;
            clk_div <= 0;
        end
        else begin
            if(cnt == div_factor/2-1) begin
                cnt <= 0;
                clk_div <= ~clk_div;
            end
            else begin
                cnt <= cnt_nxt;
            end
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module ph_fdc_logic (
    input clk,                  // clock whose frequency is going to be acquired
    input reset_n,              // reset (active low)
    input pulse,                // pulse detecting the rising edge of the divided clk_osc
    output reg [5:0] dctrl,     // control code only in FDC
    output reg [5:0] out        // control code output
);

    wire en_cnt, en_acc, en_dec, en_dec_global, rst_cnt, rst_acc_global, rst_acc_local;
    reg period, period_retimed, pulse_acc, pulse_retimed;
    reg [4:0] cnt;
    reg [3:0] n_up, n_dn, cnt_period;

    assign en_cnt = pulse;
    assign en_acc = pulse;
    assign en_dec = period;
    assign en_dec_global = pulse_acc;
    assign rst_cnt = pulse;
    assign rst_acc_global = pulse_acc;
    assign rst_acc_local = pulse_retimed;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            pulse_retimed <= 1'b0;
            period_retimed <= 1'b0;
        end
        else begin
            pulse_retimed <= pulse;
            period_retimed <= period;
        end
    end

    always @(*) begin
        pulse_acc = period&(~period_retimed);
        if(cnt_period == 4'b1110) period = 1'b1;
        else period = 1'b0;
    end 

    ph_fdc_counter fdc_counter_fd(.clk(clk), .reset_n(reset_n), .rst_sync(rst_cnt), .cnt(cnt));

    ph_fdc_counter_en fdc_counter_acc(.clk(clk), .reset_n(reset_n), .enable(en_cnt), .cnt(cnt_period));

    ph_fdc_acc fdc_acc(.clk(clk), .reset_n(reset_n), .rst_sync_global(rst_acc_global), .rst_sync_local(rst_acc_local), .enable(en_acc), .in(cnt), .n_up(n_up), .n_dn(n_dn));

    ph_fdc_decision fdc_decision(.clk(clk), .reset_n(reset_n), .enable(en_dec), .enable_global(en_dec_global), .n_up(n_up), .n_dn(n_dn), .interior_code(dctrl), .out(out));

endmodule

module ph_fdc_pulse_gen (
    input clk,
    input in,
    input reset_n,
    output reg pulse
);

    reg reg1, reg2;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            reg1 <= 1'b0;
            reg2 <= 1'b0;
        end
        else begin
            reg1 <= in;
            reg2 <= reg1;
        end
    end

    always @(*) begin
        pulse = reg1&(~reg2);
    end

endmodule

module ph_interp #(
    `parameter_realarray(f_range_cof_0), // frequency tuning coefficients for f_range=0
    `parameter_realarray(f_range_cof_1), // frequency tuning coefficients for f_range=1
    `parameter_realarray(f_range_cof_2), // frequency tuning coefficients for f_range=2
    `parameter_realarray(f_range_cof_3)  // frequency tuning coefficients for f_range=3
)(
    input vdd,
    input vss,
    input [1:0] f_range,        // operating frequency range (global)
    input [5:0] f_bias,         // bias corresponding to operating frequency
    `input_xbit clk0,			// clock input (selected when weight=0)
    `input_xbit clk1,			// clock input (selected when weight=1)
    `output_xbit clk_out,       // clock output (pos)
    `output_xbit clk_outb,	    // clock output (neg)
    input [3:0] weight,		    // thermometer-coded weight (pos)
    input [3:0] weight_b,	    // thermometer-coded weight (neg)
    input reset_n               // reset (active low)
);

    parameter real scale_factor = 1.7;
    parameter real cap = 1.0e-14;
    parameter real vth = 0.5;

    real freq;
    real delay;
    real w, wb;
    real ictrl0, ictrl1;

    xreal x1, vc1, x2, vc2, ictrl0_xr, ictrl1_xr;
    xbit qb, set, rst, pwmb, set_ex, set_ex2, sr, srb, rst_ex, rst_ex2, sr_ex, rstb;
    xbit clk0_b, clk1_b;

    always @(f_range or f_bias) begin
        case(f_range)
            2'b00 : freq = `pow(f_range_cof_0[0] + f_range_cof_0[1] * f_bias + f_range_cof_0[2] * f_bias * f_bias, 0.5) * 1e6;
            2'b01 : freq = `pow(f_range_cof_1[0] + f_range_cof_1[1] * f_bias + f_range_cof_1[2] * f_bias * f_bias, 0.5) * 1e6;
            2'b10 : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1] * f_bias + f_range_cof_2[2] * f_bias * f_bias, 0.5) * 1e6;
            2'b11 : freq = `pow(f_range_cof_3[0] + f_range_cof_3[1] * f_bias + f_range_cof_3[2] * f_bias * f_bias, 0.5) * 1e6;
            default : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1] * f_bias + f_range_cof_2[2] * f_bias * f_bias, 0.5) * 1e6;
        endcase
    end

    // interpolator delay calculation
    initial begin
        case(f_range)
            2'b00 : delay = 1/`pow(f_range_cof_0[0] + f_range_cof_0[1]*63.0 + f_range_cof_0[2]*63.0*63.0, 0.5)/1e6/8*scale_factor;
            2'b01 : delay = 1/`pow(f_range_cof_1[0] + f_range_cof_1[1]*63.0 + f_range_cof_1[2]*63.0*63.0, 0.5)/1e6/8*scale_factor;
            2'b10 : delay = 1/`pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5)/1e6/8*scale_factor;
            2'b11 : delay = 1/`pow(f_range_cof_3[0] + f_range_cof_3[1]*63.0 + f_range_cof_3[2]*63.0*63.0, 0.5)/1e6/8*scale_factor;
            default : delay = 1/`pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5)/1e6/8*scale_factor;
        endcase
    end

    always @(freq) begin
        delay = 1/freq/8*scale_factor;
    end

    always @(*) begin
        ictrl0 = cap/delay * wb * vth;
        ictrl1 = cap/delay * w * vth;
    end

    // weight calculation
    always @(*) begin
        w = (1.0*(8*weight[3]+4*weight[2]+2*weight[1]+weight[0])+0.5)/16.0;
        wb = (1.0*(8*weight_b[3]+4*weight_b[2]+2*weight_b[1]+weight_b[0])+0.5)/16.0;
    end

    // reset
    bit_to_xbit conn_b2xb0(.in(reset_n), .out(rstb));

    // phase interpolation
    real_to_xreal r2xr0(.in(ictrl0), .out(ictrl0_xr));
    real_to_xreal r2xr1(.in(ictrl1), .out(ictrl1_xr));

    inv_xbit inv_clk0(.in(clk0), .out(clk0_b));
    inv_xbit inv_clk1(.in(clk1), .out(clk1_b));

    isource #(.mode("in"))
            i1(.pos(`ground),.neg(x1), .in(ictrl0_xr));
    switch #(.R1(1e7), .R0(1e-3), .ic(0)) 
            sw_p1(.pos(x1), .neg(vc1), .ctrl(clk0));
    switch #(.R1(1e7), .R0(1e-3), .ic(0)) 
            sw_p1b(.pos(x2), .neg(vc1), .ctrl(clk1));
    switch #(.R1(1e-3), .R0(1e7), .ic(0))
            sw_n1(.pos(vc1), .neg(`ground), .ctrl(sr));

    capacitor #(.C(cap), .ic(0)) cap1(.pos(vc1), .neg(`ground));
    slice #(.threshold(vth)) slice1(.in(vc1), .in_ref(`ground), .out(set_ex));
    inv_xbit inv_set(.in(set_ex), .out(set_ex2));
    and2_xbit and2_set(.in_a(set_ex2),.in_b(rstb),.out(set));
    
    isource #(.mode("in"))
            i2(.pos(`ground),.neg(x2), .in(ictrl1_xr));
    switch #(.R1(1e7), .R0(1e-3), .ic(0))
            sw_p2(.pos(x1), .neg(vc2), .ctrl(clk0_b));
    switch #(.R1(1e7), .R0(1e-3), .ic(0))
            sw_p2b(.pos(x2), .neg(vc2), .ctrl(clk1_b));
    switch #(.R1(1e-3), .R0(1e7), .ic(0))
            sw_n2(.pos(vc2), .neg(`ground), .ctrl(srb));
    capacitor #(.C(cap), .ic(0)) cap2(.pos(vc2), .neg(`ground));

    slice #(.threshold(vth)) slice2(.in(vc2), .in_ref(`ground), .out(rst_ex));
    buf_xbit buf_rst(.in(rst_ex), .out(rst_ex2));
    nand2_xbit nand2_rst(.in_a(rst_ex2),.in_b(rstb),.out(rst));

    // SR latch
    nand2_xbit nand_rslatch0(.in_a(set), .in_b(srb), .out(sr_ex));
    nand2_xbit nand_rslatch1(.in_a(rst), .in_b(sr_ex), .out(srb));

    inv_xbit inv1(.in(sr),.out(clk_out));
    inv_xbit inv2(.in(srb),.out(clk_outb));
    mux_xbit mux1(.in({sr_ex,`one_xbit}),.sel(rstb),.out(sr));

endmodule

module ph_mixer #(
    `parameter_realarray(f_range_cof_0), // frequency tuning coefficients for f_range=0
    `parameter_realarray(f_range_cof_1), // frequency tuning coefficients for f_range=1
    `parameter_realarray(f_range_cof_2), // frequency tuning coefficients for f_range=2
    `parameter_realarray(f_range_cof_3)  // frequency tuning coefficients for f_range=3
)(
    input vdd,
    input vss,
    input reset_n,              // reset (active low)
    input [1:0] f_range,        // operating frequency range (globa)
    input [5:0] f_bias,         // bias corresponding to operating frequency
    `input_xbit [7:0] clk_in,   // multiphase clock inputs
    input [3:0] sel_even,       // phase mux selection inputs (even)
    input [3:0] sel_odd,        // phase mux selection inputs (odd)
    input [3:0] weight,         // phase interpolator weight control
    `output_xbit piclk,         // interpolated clock output (pos)
    `output_xbit piclkb         // interpolated clock output (neg)
);

    wire [3:0] weightb;
    xbit [1:0] clk_mux;
    xbit [3:0] sel_even_xbit;
    xbit [3:0] sel_odd_xbit;

    assign weightb = ~weight;

    // phase multiplexers
    bit_to_xbit #(.width(4)) conn_even(.in(sel_even), .out(sel_even_xbit));
    bit_to_xbit #(.width(4)) conn_odd(.in(sel_odd), .out(sel_odd_xbit));

    ph_mux ph_mux_even (
        .in({clk_in[6],clk_in[4],clk_in[2],clk_in[0]}), .out(clk_mux[0]), .sel(sel_even_xbit)
    );
    ph_mux ph_mux_odd (
        .in({clk_in[7],clk_in[5],clk_in[3],clk_in[1]}), .out(clk_mux[1]), .sel(sel_odd_xbit)
    );

    // phase interpolator
    ph_interp #(.f_range_cof_0(f_range_cof_0),
                .f_range_cof_1(f_range_cof_1),
                .f_range_cof_2(f_range_cof_2),
                .f_range_cof_3(f_range_cof_3))
        ph_interp(.vdd(vdd), .vss(vss),
                  .f_range(f_range), .f_bias(f_bias),
                  .clk0(clk_mux[0]), .clk1(clk_mux[1]),
                  .clk_out(piclk), .clk_outb(piclkb),
                  .weight(weight), .weight_b(weightb),
                  .reset_n(reset_n)
    );

endmodule

module ph_mux (
    `input_xbit [3:0] in,   // clock inputs
    `output_xbit out,       // clock output
    `input_xbit [3:0] sel   // selection control (one-hot encoded)
);

    // 4:1 multiplexer
    tribuf_xbit tribuf0(.en(sel[0]), .in(in[0]), .out(out));
    tribuf_xbit tribuf1(.en(sel[1]), .in(in[1]), .out(out));
    tribuf_xbit tribuf2(.en(sel[2]), .in(in[2]), .out(out));
    tribuf_xbit tribuf3(.en(sel[3]), .in(in[3]), .out(out));

endmodule

module ph_mux_enc (
    input vdd,
    input vss,
    input clk,                  // clock input (positive)
    input clkb,                 // clock input (negative)
    input reset_n,              // reset (active low)
    input [6:0] sel_in,         // mux selection signal inputs
    output [11:0] sel_out       // mux selection signal outputs
);
    
    reg [6:0] reg1;
    wire [11:0] sel_enc;
    reg [11:0] reg2;

    // first-level registers for retiming
    always @(posedge clk or negedge reset_n) begin
        if (!reset_n) reg1 <= 7'b0;
        else reg1 <= sel_in;
    end

    // encoder logic (must be placed after retiming)
    assign sel_enc[3:0] = reg1[4] ? ~reg1[3:0] : reg1[3:0];
    assign sel_enc[7:4] = {reg1[6],reg1[6],~reg1[6],~reg1[6]}&{reg1[5],~reg1[5],reg1[5],~reg1[5]};
    assign sel_enc[11:8] = reg1[4] ? {sel_enc[6:4],sel_enc[7]} : sel_enc[7:4];

    // second-level registers for retiming
    always @(posedge clk or negedge reset_n) begin
        if (!reset_n) reg2 <= 12'b0001_0001_0000;
        else reg2 <= sel_enc;
    end

    // final output
    assign sel_out = reg2;

endmodule

module ph_relaxation_osc #(
    `parameter_real(scale_factor, 2),           // frequency multiplication
    `parameter_realarray(f_range_cof_0), // frequency tuning coefficients for f_range=0
    `parameter_realarray(f_range_cof_1), // frequency tuning coefficients for f_range=1
    `parameter_realarray(f_range_cof_2), // frequency tuning coefficients for f_range=2
    `parameter_realarray(f_range_cof_3)  // frequency tuning coefficients for f_range=3
)(
    input vdd,
    input vss,
    input [1:0] f_range,        // operation frequency range
    input [5:0] fdc_in,         // code converted from frequency
    input reset_n,              // reset (active low)
    output clk_relax_osc        // oscillation frequnecy
);

    real freq;
    xreal freq_xr;
    xbit clk_relax_osc_xbit;

    always @(*) begin
        if($realtime != 0) begin
            if(!reset_n) begin
                freq = 0;
            end
            else begin
                case(f_range)
                    2'b00 : freq = scale_factor * `pow(f_range_cof_0[0] + f_range_cof_0[1] * fdc_in + f_range_cof_0[2] * fdc_in * fdc_in, 0.5) * 1e6;
                    2'b01 : freq = scale_factor * `pow(f_range_cof_1[0] + f_range_cof_1[1] * fdc_in + f_range_cof_1[2] * fdc_in * fdc_in, 0.5) * 1e6;
                    2'b10 : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1] * fdc_in + f_range_cof_2[2] * fdc_in * fdc_in, 0.5) * 1e6;
                    2'b11 : freq = scale_factor * `pow(f_range_cof_3[0] + f_range_cof_3[1] * fdc_in + f_range_cof_3[2] * fdc_in * fdc_in, 0.5) * 1e6;
                    default : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1] * fdc_in + f_range_cof_2[2] * fdc_in * fdc_in, 0.5) * 1e6;
                endcase
            end
        end
        else begin
            case(f_range)
                2'b00 : freq = scale_factor * `pow(f_range_cof_0[0] + f_range_cof_0[1]*63.0 + f_range_cof_0[2]*63.0*63.0, 0.5) * 1e6;
                2'b01 : freq = scale_factor * `pow(f_range_cof_1[0] + f_range_cof_1[1]*63.0 + f_range_cof_1[2]*63.0*63.0, 0.5) * 1e6;
                2'b10 : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5) * 1e6;
                2'b11 : freq = scale_factor * `pow(f_range_cof_3[0] + f_range_cof_3[1]*63.0 + f_range_cof_3[2]*63.0*63.0, 0.5) * 1e6;
            endcase
        end
    end

    real_to_xreal conn0(.in(freq), .out(freq_xr));
    freq_to_clk freq2clk(.in(freq_xr), .out(clk_relax_osc_xbit));
    xbit_to_bit conn1(.in(clk_relax_osc_xbit), .out(clk_relax_osc));

endmodule

module ph_rotator_core #(
    `parameter_realarray(f_range_cof_0), // frequency tuning coefficients for f_range=0
    `parameter_realarray(f_range_cof_1), // frequency tuning coefficients for f_range=1
    `parameter_realarray(f_range_cof_2), // frequency tuning coefficients for f_range=2
    `parameter_realarray(f_range_cof_3)  // frequency tuning coefficients for f_range=3
)(
    input vdd,
    input vss,
    input reset_n,              // reset (active low)
    `input_xbit [7:0] clk_in,   // multiphase clock inputs
    input [6:0] phsel,          // phase select input
    input [1:0] f_range,        // frequency range
    input [5:0] f_bias,         // frequency calibration code
    input [5:0] fdc_ctrl,       // control code of relaxation osc for FDC
    `output_xbit clk_out,       // clock output
    `output_xbit clk_outb,      // clock output (negative)
    output reg clk_fdc,         // clock for FDC
    output reg pulse            // pulse signal for FDC
);

    reg clk_in_bit;
    wire pi_sync_clk, pi_sync_clkb; // internal clock for timing synchorinization (caution : synthesis)
    wire [3:0] weight;
    wire [3:0] sel_even, sel_odd;
    wire [11:0] sel_out;

    // encoder and retimers for selection signals (digital)
    ph_mux_enc ph_mux_enc (
        .vdd(vdd), .vss(vss),
        .clk(pi_sync_clk), .clkb(pi_sync_clkb), .reset_n(reset_n),
        .sel_in(phsel), .sel_out(sel_out));

    assign weight = sel_out[3:0];
    assign sel_odd = sel_out[7:4];
    assign sel_even = sel_out[11:8];

    // phase multiplexers and interpolator (analog)
    ph_mixer #(.f_range_cof_0(f_range_cof_0),
               .f_range_cof_1(f_range_cof_1),
               .f_range_cof_2(f_range_cof_2),
               .f_range_cof_3(f_range_cof_3))
        ph_mixer (.vdd(vdd), .vss(vss),
                  .clk_in(clk_in), .reset_n(reset_n),
                  .f_range(f_range), .f_bias(f_bias),
                  .sel_even(sel_even), .sel_odd(sel_odd), .weight(weight),
                  .piclk(clk_out), .piclkb(clk_outb));

    xbit_to_bit conn0(.in(clk_out), .out(pi_sync_clk));
    assign pi_sync_clkb = ~pi_sync_clk;

    // relaxation oscillator and pulse generator for the frequency to digital converter (frequency auto calibration for constant Ktdc)
    ph_relaxation_osc #(.scale_factor(2),
                        .f_range_cof_0(f_range_cof_0),
                        .f_range_cof_1(f_range_cof_1),
                        .f_range_cof_2(f_range_cof_2),
                        .f_range_cof_3(f_range_cof_3))
            ph_relaxation_osc(.vdd(vdd), .vss(vss),
                    .f_range(f_range), .fdc_in(fdc_ctrl),
                    .reset_n(reset_n), .clk_relax_osc(clk_osc));

    xbit_to_bit conn(.in(clk_in[0]), .out(clk_in_bit));

    ph_fdc_controller ph_fdc_controller (
                        .f_range(f_range), .clk_in(clk_in_bit),
                        .clk_osc(clk_osc), .reset_n(reset_n),
                        .clk_fdc(clk_fdc), .pulse(pulse));

endmodule
/*----------------------------------------------------------------------
MODEL div8_mphase.sv

= Purpose =

A divided-by-8 block that generates multi-phase outputs

= Description =

The block consists of 4 differential flip-flops in ring configuration.

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

module div8_mphase #(
    `parameter_real(delay_cq, 242e-12),     // clock-to-output delay
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 2.1e4)         // equivalent output resistance
)(
    input vdd,
    input vss,
    `input_xbit clk_inp,        // clock input(pos)
    `input_xbit clk_inn,        // clock input(neg)
    input reset_n,              // divider reset (active low)
    `output_xbit [7:0] clk_out  // divided clock outputs
);

    xbit rst;
    xbit div0, divb0;

    bit_to_xbit conn0(.in(~reset_n), .out(rst));

    dff_rst_async_xbit #(.delay_cq(delay_cq))
        dff0(.d(clk_out[7]), .q(div0), .clk(clk_inp), .rst(rst));
    dff_rst_async_xbit #(.delay_cq(delay_cq))
        dff1(.d(clk_out[6]), .q(clk_out[7]), .clk(clk_inp), .rst(rst));
    dff_rst_async_xbit #(.delay_cq(delay_cq))
        dff2(.d(clk_out[5]), .q(clk_out[6]), .clk(clk_inp), .rst(rst));
    dff_rst_async_xbit #(.delay_cq(delay_cq))
        dff3(.d(clk_out[4]), .q(clk_out[5]), .clk(clk_inp), .rst(rst));
    inv_xbit    inv0(.in(div0), .out(clk_out[4]));

    dff_set_async_xbit #(.delay_cq(delay_cq))
        dff0_b(.d(clk_out[3]), .q(divb0), .clk(clk_inn), .set(rst));
    dff_set_async_xbit #(.delay_cq(delay_cq))
        dff1_b(.d(clk_out[2]), .q(clk_out[3]), .clk(clk_inn), .set(rst));
    dff_set_async_xbit #(.delay_cq(delay_cq))
        dff2_b(.d(clk_out[1]), .q(clk_out[2]), .clk(clk_inn), .set(rst));
    dff_set_async_xbit #(.delay_cq(delay_cq))
        dff3_b(.d(clk_out[0]), .q(clk_out[1]), .clk(clk_inn), .set(rst));
    inv_xbit    inv1(.in(divb0), .out(clk_out[0]));

endmodule
/*----------------------------------------------------------------------
MODEL divider.sv

= Purpose =

A clock frequency divider with a parameterized division factor.

= Description =

This divider is triggered by both the rising and falling edges of the input clock in order to keep the output duty cycle close to 50% even for odd division factors. For the dividing factor, 2 to 9 are available.

A fixed-timestep model.

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module divider #(
    `parameter_real(delay_cq, 275e-12),     // clock-to-output delay
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 2.1e4)         // equivalent output resistance
)(
    input vdd,
    input vss,
    input reset_n,          // reset signal (active low)
    input [2:0] div_sel,    // dividing factor selection code (2,3,4,5,6,7,8,9)
    `input_xbit clk_inp,    // input positive clock
    `input_xbit clk_inn,    // input negative clock
    `output_xbit clk_outp,  // output positive clock 
    `output_xbit clk_outn   // output negative clock 
);

    wire [3:0] next;
    wire enable;
    reg [2:0] div_sel_retimed;
    reg [3:0] div_factor;
    reg [3:0] count;

    xbit rst, en;
    xbit d0;
//    xbit out0, out1;
    
    always @(posedge clk_outp or negedge reset_n) begin
        if(!reset_n) begin
            div_sel_retimed <= 3'b000;
        end    
        else begin
            div_sel_retimed <= div_sel;
        end
    end

    always @(*) begin
        case(div_sel_retimed)
            3'b000 : div_factor = 1;    // dividing factor = 2
            3'b001 : div_factor = 2;    // dividing factor = 3
            3'b010 : div_factor = 3;    // dividing factor = 4
            3'b011 : div_factor = 4;    // dividing factor = 5
            3'b100 : div_factor = 5;    // dividing factor = 6
            3'b101 : div_factor = 6;    // dividing factor = 7
            3'b110 : div_factor = 7;    // dividing factor = 8
            3'b111 : div_factor = 8;    // dividing factor = 9
            default : div_factor = 1;   // dividing factor = 2
        endcase
    end

    assign next = (count == div_factor) ? 0 : (count + 1);
    assign enable = (count == div_factor) ? 1'b1 : 1'b0;

    always @(posedge clk_inp or posedge clk_inn or negedge reset_n) begin
        if (!reset_n) begin
            count <= 4'b1111;
        end
        else begin
            #(delay_cq/`TIME_SCALE) count <= next;
        end
    end

    bit_to_xbit conn0(.in(enable), .out(en));
    bit_to_xbit conn1(.in(~reset_n), .out(rst));

    div_dff_dual_edge #(.delay_cq(delay_cq))
            dff_dual_edge(.d(d0), .rst(rst),
                .clkp(clk_inp), .clkn(clk_inn), .q(clk_outn));

    inv_xbit inv0(.in(clk_outn), .out(clk_outp));

    mux_xbit mux0(.in({clk_outp,clk_outn}), .sel(en), .out(d0));

endmodule

module div_dff_dual_edge #(
    `parameter_real(delay_cq, 100e-12)  // clk-to-q delay
)(
    `input_xbit d,
    `input_xbit rst,
    `input_xbit clkp,
    `input_xbit clkn,
    `output_xbit q
);

    xbit q0, q1;
    xbit clkn_b, clk_or;

    dff_rst_async_xbit  #(.delay_cq(0.0))
                dff0(.d(d), .clk(clkp), .rst(rst), .q(q0));
    dff_rst_async_xbit  #(.delay_cq(0.0))
                dff1(.d(d), .clk(clkn), .rst(rst), .q(q1));

    inv_xbit    inv0(.in(clkn), .out(clkn_b));

    or_xbit    or0(.in({clkp,clkn_b}), .out(clk_or));

    mux_xbit #(.delay(delay_cq))
                mux0(.in({q0,q1}), .sel(clk_or), .out(q));

endmodule
/*----------------------------------------------------------------------
MODEL prescaler.sv

= Purpose =

A prescaler with a parameterized division factor.

= Description =

This prescaler is triggered by both the rising and falling edges of the input clock in order to keep the output duty cycle close to 50% even for odd division factors. For the dividing factor, 2, 4, and 8 are available.

A fixed-timestep model.

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module prescaler #(
    `parameter_real(delay_cq_0, 225e-12), // clock-to-output delay 0
    `parameter_real(delay_cq_1, 332e-12), // clock-to-output delay 1
    `parameter_real(delay_cq_2, 405e-12), // clock-to-output delay 2
    `parameter_real(cap_in, 1.8e-15),     // equivalent input capacitance
    `parameter_real(res_out, 0.52e4)      // equivalent output resistance
)(
    input vdd,
    input vss,
    input reset_n,          // reset signal (active low)
    input [1:0] div_sel,    // dividing factor selection code (2,4,8)
    `input_xbit clk_inp,    // input clock (positive)
    `input_xbit clk_inn,    // input clock (negative)
    `output_xbit clk_outp,  // output clock (positive)
    `output_xbit clk_outn   // output clock (negative)
);

    reg [1:0] div_sel_retimed;
    xbit [1:0] div_sel_xbit;
    xbit rst;
    xbit clk_div2, clk_div2_n, clk_div4, clk_div4_n, clk_div8, clk_div8_n;
    xbit clk_div2_out, clk_div4_out, clk_div8_out;
    xbit clk_div2_out_n, clk_div4_out_n, clk_div8_out_n;

    always @(posedge clk_outp or negedge reset_n) begin
        if(!reset_n) begin
            div_sel_retimed <= 2'b00;
        end
        else begin
            div_sel_retimed <= div_sel;
        end
    end

    bit_to_xbit #(.width(2)) conn0(.in(div_sel_retimed), .out(div_sel_xbit));
    bit_to_xbit conn1(.in(~reset_n), .out(rst));

    dff_rst_async_xbit
        dff0(.d(clk_div2), .q(clk_div2_n), .clk(clk_inp), .rst(rst));
    inv_xbit inv0(.in(clk_div2_n), .out(clk_div2));

    dff_rst_async_xbit
        dff1(.d(clk_div4), .q(clk_div4_n), .clk(clk_div2_n), .rst(rst));
    inv_xbit inv1(.in(clk_div4_n), .out(clk_div4));

    dff_rst_async_xbit
        dff2(.d(clk_div8), .q(clk_div8_n), .clk(clk_div4_n), .rst(rst));
    inv_xbit inv2(.in(clk_div8_n), .out(clk_div8));

    delay_xbit #(.delay(delay_cq_0))
        delay_0(.in(clk_div2), .out(clk_div2_out));
    delay_xbit #(.delay(delay_cq_0))
        delay_0b(.in(clk_div2_n), .out(clk_div2_out_n));
    delay_xbit #(.delay(delay_cq_1))
        delay_1(.in(clk_div4), .out(clk_div4_out));
    delay_xbit #(.delay(delay_cq_1))
        delay_1b(.in(clk_div4_n), .out(clk_div4_out_n));
    delay_xbit #(.delay(delay_cq_2))
        delay_2(.in(clk_div8), .out(clk_div8_out));
    delay_xbit #(.delay(delay_cq_2))
        delay_2b(.in(clk_div8_n), .out(clk_div8_out_n));

    mux_xbit #(.num_in(4), .width_sel(2))
        mux0(.sel(div_sel_xbit), .in({clk_div8_out,clk_div8_out,clk_div4_out,clk_div2_out}), .out(clk_outp));
    mux_xbit #(.num_in(4), .width_sel(2))
        mux1(.sel(div_sel_xbit), .in({clk_div8_out_n,clk_div8_out_n,clk_div4_out_n,clk_div2_out_n}), .out(clk_outn));

endmodule
/*----------------------------------------------------------------------
MODEL ring_dco.sv

= Purpose =

A ring-based DCO block (custom analog).

= Description =

A fixed-timestep model.

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module ring_dco #(
    `parameter_real(K_dco, 0.00056),        // DCO relative gain
    `parameter_real(freq_center, 1.41e9),   // center of the DCO output frequency
    `parameter_real(PN_foffset, 1e6),       // phase noise frequency offset
    `parameter_real(PN_dbc, -85),           // phase noise at the frequency offset
    `parameter_real(delay_in_int, 512e-12), // integral path input delay
    `parameter_real(delay_in_prop, 121e-12),// proportional path input delay
    `parameter_real(delay_cq, 322e-12),     // clock-to-output delay
    `parameter_real(t_const_int, 224e-12),  // integral path time-constant
    `parameter_real(t_const_prop, 35e-12),  // proportional path time-constant
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 0.26e4)        // equivalent output resistance
)(
    input vdd,
    input vss,
    input clk_lf,               // input retiming clock
    input [2:0] dco_pgain,      // DCO proportional gain control (binary-coded)
    `input_xbit prop_up,        // proportional control (up, binary-coded)
    `input_xbit prop_dn,        // proportional control (dn, binary-coded)
    input [9:0] in,             // DCO control input (binary-coded)
    input signed [2:0] in_dsm,  // DCO control DSM input (binary-coded)
    input reset_n,	            // DCO start-up signal for current reference
    `output_xbit [7:0] clk_out  // uniformly-spaced multiphase output clock
);

    parameter PN_fcenter = freq_center;

    reg [9:0] in_delay;
    reg signed [2:0] in_dsm_delay;
    reg [30:0] int_coarse, int_fine;    // coarse & fine code (thermometer-coded)
    reg [2:0] int_dsm;                  // DSM code (binary-coded)
    reg [6:0] prop_gain;

    xbit [9:0] in_xbit, in_delay_xbit;
    xbit [2:0] in_dsm_xbit, in_dsm_delay_xbit;

    bit_to_xbit #(.width(10)) conn_in(.in(in), .out(in_xbit));
    bit_to_xbit #(.width(3)) conn_in_dsm(.in(in_dsm), .out(in_dsm_xbit));

    genvar j;
    generate 
        for (j=0; j<10; j=j+1) begin : gen_loop_in
        delay_xbit #(.delay(delay_in_int))
            delay_xbit_in(.in(in_xbit[j]), .out(in_delay_xbit[j]));
        end
        for (j=0; j<3; j=j+1) begin : gen_loop_in_dsm
        delay_xbit #(.delay(delay_in_int))
            delay_xbit_in_dsm(.in(in_dsm_xbit[j]), .out(in_dsm_delay_xbit[j]));
        end
    endgenerate
 
    xbit_to_bit #(.width(10)) conn_in_delay(.in(in_delay_xbit), .out(in_delay));
    xbit_to_bit #(.width(3)) conn_in_dsm_delay(.in(in_dsm_delay_xbit), .out(in_dsm_delay));
  
    // binary to thermometer encode with retimer
    ring_dco_encoder therm_encoder (
            .clk(clk_lf), .reset_n(reset_n),
            .in(in_delay), .in_dsm(in_dsm_delay), .dco_pgain(dco_pgain),
            .int_coarse(int_coarse), .int_fine(int_fine), .int_dsm(int_dsm),
            .prop_gain(prop_gain)
        );
    // core ring_dco instances
    ring_dco_core #(.num_phase(8),
        .int_coarse_steps(31), .int_fine_steps(31), .int_dsm_steps(3),
        .freq_center(freq_center), .K_dco(K_dco),
        .PN_fcenter(PN_fcenter), .PN_foffset(PN_foffset), .PN_dbc(PN_dbc),
        .delay_in_int(delay_cq), .delay_in_prop(delay_in_prop),
        .t_const_int(t_const_int), .t_const_prop(t_const_prop))
        ring_dco_core (.vdd(vdd), .vss(vss), .reset_n(reset_n),
                .prop_gain(prop_gain), .prop_up(prop_up), .prop_dn(prop_dn),
                .int_coarse(int_coarse), .int_fine(int_fine), .int_dsm(int_dsm),
                .out(clk_out));

endmodule

module ring_dco_bin_encoder_dsm( 
    input [2:0] din,        // binary-coded input
    output reg [2:0] dout   // thermometer-coded output
);

    // binary-to-thermometer encoder
    always @(din) begin
        case(din)
            3'b100 : dout = 3'b111;
            3'b101 : dout = 3'b111;
            3'b110 : dout = 3'b110;
            3'b111 : dout = 3'b101;
            3'b000 : dout = 3'b000;
            3'b001 : dout = 3'b001;
            3'b010 : dout = 3'b010;
            3'b011 : dout = 3'b011;
        endcase
    end

endmodule

module ring_dco_core #(
    `parameter_integer(num_phase, 8),           // number of output phases
    `parameter_integer(int_coarse_steps, 31),   // integral control bits (thermometer/coarse)
    `parameter_integer(int_fine_steps, 31),     // integral control bits (thermometer/fine)
    `parameter_integer(int_dsm_steps, 7),       // integral control bits (thermometer/DSM)
    `parameter_real(K_dco, 0.00056),        // DCO relative gain
    `parameter_real(freq_center, 1.5e9),    // center of the DCO output frequency
    `parameter_real(init_phase, M_PI/2),    // initial phase (randomized if <0)
    `parameter_real(RJ_kappa, 0.0e-7),      // RMS random jitter after 1-second free-running
    `parameter_real(PN_fcenter, 1.5e9),     // phase noise center frequency
    `parameter_real(PN_foffset, 1.0e6),     // phase noise offset frequency
    `parameter_real(PN_dbc, -85),           // phase noise measured in dBc
    `parameter_real(delay_in_int, 150.0e-12),   // integral path input delay
    `parameter_real(delay_in_prop, 200.0e-12),  // proportional path input delay
    `parameter_real(t_const_int, 100.0e-12),    // integral path time-constant
    `parameter_real(t_const_prop, 40.0e-12)     // proportional path time-constant
)(
    input vdd,
    input vss,
    input reset_n,                          // reset (active-low)
    `input_xbit prop_up,                    // proportional up input pulse bit
    `input_xbit prop_dn,                    // proportional down input pulse bit
    input [int_coarse_steps-1:0] int_coarse,// integral input code (thermometer/coarse)
    input [int_fine_steps-1:0] int_fine,    // integral input code (thermometer/fine)
    input [int_dsm_steps-1:0] int_dsm,      // DSM input code (thermometer-coded)
    input [6:0] prop_gain,                  // DCO proportional gain control (thermometer-coded)
    `output_xbit [num_phase-1:0] out        // output clocks
);

    // parameters
    parameter real BW_int = 1/t_const_int;
    parameter real BW_prop = 1/t_const_prop;

    // variables
    real prop_gain_r;
    real freq;
    xreal freq_xr;

    bit [num_phase-1:0] out_init;
    xbit [num_phase-1:0] out_init_xbit;
    xbit [num_phase-1:0] out_xbit;
    xbit rstn;

    xreal int_ctrl_xr, dsm_ctrl_xr, prop_up_xr, prop_dn_xr;
    xreal int_ctrl_filter, dsm_ctrl_filter, prop_up_filter, prop_dn_filter;
    xreal int_factor, prop_factor;
    real int_factor_r, prop_factor_r;

    integer i;

    // DCO proportional gain control
    initial begin
        for(i=0; i<num_phase; i=i+1) begin
            if(((i+1)/2 - i/2) == 1) out_init[i] = 1'b1;
            else out_init[i] = 1'b0;
        end
    end

    bit_to_xbit #(.width(num_phase)) conn0(.in(out_init), .out(out_init_xbit));

    always @(prop_gain) begin
        case (prop_gain)
            7'b0000001 : prop_gain_r = 0.5*32;
            7'b0000011 : prop_gain_r = 1*32;
            7'b0000111 : prop_gain_r = 2*32;
            7'b0001111 : prop_gain_r = 4*32;
            7'b0011111 : prop_gain_r = 8*32;
            7'b0111111 : prop_gain_r = 16*32;
            7'b1111111 : prop_gain_r = 32*32;
            default : prop_gain_r = 0*32;
        endcase
    end

    real int_ctrl, dsm_ctrl;

    always @(int_coarse or int_fine or int_dsm) begin
        // integral contrl
        int_ctrl = 0.0;
        dsm_ctrl = 0.0;
        for (i = 0; i < int_coarse_steps; i=i+1)
            int_ctrl = int_ctrl+int_coarse[i];
        int_ctrl = int_ctrl*(int_coarse_steps + 1);

        for (i = 0; i < int_fine_steps; i=i+1)
            int_ctrl = int_ctrl+int_fine[i];

        if(int_dsm[int_dsm_steps-1])
            dsm_ctrl = -int_dsm[int_dsm_steps-2:0];
        else
            dsm_ctrl = int_dsm[int_dsm_steps-2:0];
    end

    real_to_xreal conn_int(.in(int_ctrl), .out(int_ctrl_xr));
    real_to_xreal conn_dsm(.in(dsm_ctrl), .out(dsm_ctrl_xr));
    transition tran_pup(.in(prop_up), .out(prop_up_xr));
    transition tran_pdn(.in(prop_dn), .out(prop_dn_xr));

    filter #(.poles({BW_int,0.0}), .zeros({0}), .delay(delay_in_int))
            filter_int(.in(int_ctrl_xr), .out(int_ctrl_filter));

    filter #(.poles({BW_int,0.0}), .zeros({0}), .delay(delay_in_int))
            filter_dsm(.in(dsm_ctrl_xr), .out(dsm_ctrl_filter));

    filter_var #(.delay(delay_in_prop), .num_poles(1), .num_zeros(0))
            filter_prop_up(.in(prop_up_xr), .out(prop_up_filter),
                           .poles({BW_prop,0.0}), .zeros(),
                           .gain(prop_gain_r));
    filter_var #(.delay(delay_in_prop), .num_poles(1), .num_zeros(0))
            filter_prop_dn(.in(prop_dn_xr), .out(prop_dn_filter),
                           .poles({BW_prop,0.0}), .zeros(),
                           .gain(prop_gain_r));

    poly_func #(.num_in(2), .data({-(int_coarse_steps+1)*(int_fine_steps+1)/2.0,1.0,1.0})) poly_func0(.in({int_ctrl_filter,dsm_ctrl_filter}), .out(int_factor));
    add #(.scale({K_dco,-K_dco})) add0(.in({prop_up_filter,prop_dn_filter}), .out(prop_factor));

    xreal_to_real conn_int_factor(.in(int_factor), .out(int_factor_r));
    xreal_to_real conn_prop_factor(.in(prop_factor), .out(prop_factor_r));

    // digital-to-frequency conversion (exponential D-to-f)
    always @(int_factor_r or prop_factor_r) begin
        freq = freq_center * `pow(1 + K_dco, int_factor_r) * (1 + prop_factor_r);
    end

    real_to_xreal conn1(.in(freq), .out(freq_xr));

    // frequency-to-clock conversion
    freq_to_clk #(.init_phase(init_phase), .num_phase(num_phase),
                .RJ_kappa(RJ_kappa), .PN_foffset(PN_foffset),
                .PN_fcenter(PN_fcenter), .PN_dbc(PN_dbc))
            freq2clk(.in(freq_xr), .out(out_xbit));

    bit_to_xbit conn2(.in(reset_n), .out(rstn));

    genvar j;
    generate
        for (j=0; j<num_phase; j=j+1) begin : gen_loop
        mux_xbit out_gen(.in({out_xbit[j],out_init_xbit[j]}), .sel(rstn), .out(out[j]));
        end
    endgenerate

endmodule

module ring_dco_encoder(
    input clk,                  // input retiming clock
    input reset_n,              // reset (active-low)
    input [9:0] in,             // binary-coded integral path input
    input signed [2:0] in_dsm,  // binary-coded DSM input (signed)
    input [2:0] dco_pgain,      // bianry-coded proportional path gain control code
    output reg [30:0] int_coarse,   // thermometer-coded integral code (coarse)
    output reg [30:0] int_fine,     // thermometer-coded integral code (fine)
    output reg [2:0] int_dsm,       // binary-coded integral code (DSM)
    output reg [6:0] prop_gain      // thermometer-ocded proportional path gain control code
);

    reg [9:0] in_retimed;
    reg [2:0] in_dsm_retimed;
    reg [2:0] dco_pgain_retimed;

    // binary to thermometer decoder
    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            in_retimed <= 10'b10_0000_0000;
            in_dsm_retimed <= 3'b000;
            dco_pgain_retimed <= 3'b000;
        end
        else begin
            in_retimed <= in;
            in_dsm_retimed <= in_dsm;
            dco_pgain_retimed <= dco_pgain;
        end
    end

    ring_dco_therm_encoder therm_coarse(.din(in_retimed[9:5]), .dout(int_coarse));
    ring_dco_therm_encoder therm_fine(.din(in_retimed[4:0]), .dout(int_fine));

    ring_dco_bin_encoder_dsm bin_dsm(.din(in_dsm_retimed), .dout(int_dsm));
    ring_dco_therm_encoder_pgain therm_pgain(.din(dco_pgain_retimed), .dout(prop_gain));

endmodule

module ring_dco_therm_encoder( 
    input [4:0] din,        // binary-coded input
    output reg [30:0] dout  // thermometer-coded output
);

    // binary-to-thermometer encoder
    always @(din) begin
        case(din)
            5'b00000 : dout = 31'b0000000000000000000000000000000;
            5'b00001 : dout = 31'b0000000000000000000000000000001;
            5'b00010 : dout = 31'b0000000000000000000000000000011;
            5'b00011 : dout = 31'b0000000000000000000000000000111;
            5'b00100 : dout = 31'b0000000000000000000000000001111;
            5'b00101 : dout = 31'b0000000000000000000000000011111;
            5'b00110 : dout = 31'b0000000000000000000000000111111;
            5'b00111 : dout = 31'b0000000000000000000000001111111;

            5'b01000 : dout = 31'b0000000000000000000000011111111;
            5'b01001 : dout = 31'b0000000000000000000000111111111;
            5'b01010 : dout = 31'b0000000000000000000001111111111;
            5'b01011 : dout = 31'b0000000000000000000011111111111;
            5'b01100 : dout = 31'b0000000000000000000111111111111;
            5'b01101 : dout = 31'b0000000000000000001111111111111;
            5'b01110 : dout = 31'b0000000000000000011111111111111;
            5'b01111 : dout = 31'b0000000000000000111111111111111;

            5'b10000 : dout = 31'b0000000000000001111111111111111;
            5'b10001 : dout = 31'b0000000000000011111111111111111;
            5'b10010 : dout = 31'b0000000000000111111111111111111;
            5'b10011 : dout = 31'b0000000000001111111111111111111;
            5'b10100 : dout = 31'b0000000000011111111111111111111;
            5'b10101 : dout = 31'b0000000000111111111111111111111;
            5'b10110 : dout = 31'b0000000001111111111111111111111;
            5'b10111 : dout = 31'b0000000011111111111111111111111;

            5'b11000 : dout = 31'b0000000111111111111111111111111;
            5'b11001 : dout = 31'b0000001111111111111111111111111;
            5'b11010 : dout = 31'b0000011111111111111111111111111;
            5'b11011 : dout = 31'b0000111111111111111111111111111;
            5'b11100 : dout = 31'b0001111111111111111111111111111;
            5'b11101 : dout = 31'b0011111111111111111111111111111;
            5'b11110 : dout = 31'b0111111111111111111111111111111;
            5'b11111 : dout = 31'b1111111111111111111111111111111;
        endcase
    end

endmodule

module ring_dco_therm_encoder_pgain( 
    input [2:0] din,        // binary-coded input
    output reg [6:0] dout   // thermometer-coded output
);

    // binary-to-thermometer encoder
    always @(din) begin
        case(din)
            3'b000 : dout = 7'b0000000;
            3'b001 : dout = 7'b0000001;
            3'b010 : dout = 7'b0000011;
            3'b011 : dout = 7'b0000111;
            3'b100 : dout = 7'b0001111;
            3'b101 : dout = 7'b0011111;
            3'b110 : dout = 7'b0111111;
            3'b111 : dout = 7'b1111111;
        endcase
    end

endmodule
/*----------------------------------------------------------------------
MODEL lcdco.sv

= Purpose =

A LCDCO block (custom analog).

= Description =

A fixed-timestep model.

= Revisions =

$Authors$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module lcdco #(
    `parameter_real(K_dco, 0.00010),        // DCO relative gain
    `parameter_real(freq_center, 8.0e9),    // center of the DCO output frequency
    `parameter_real(PN_foffset, 1e6),       // phase noise frequency offset
    `parameter_real(PN_dbc, -112),          // phase noise at the frequency offset
    `parameter_real(delay_in_int, 440e-12), // integral path input delay
    `parameter_real(delay_in_prop, 242e-12),// proportional path input delay
    `parameter_real(delay_cq, 322e-12),     // clock-to-output delay
    `parameter_real(t_const_int, 102e-12),  // integral path time-constant
    `parameter_real(t_const_prop, 41e-12),  // proportional path time-constant
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 0.26e4)        // equivalent output resistance
)(
    input vdd,
    input vss,
    input clk_lf,               // input retiming clock
    input [2:0] dco_pgain,      // DCO proportional gain control (binary-coded)
    `input_xbit prop_up,        // proportional control (up, binary-coded)
    `input_xbit prop_dn,        // proportional control (dn, binary-coded)
    input [9:0] in,             // DCO control input (binary-coded)
    input signed [2:0] in_dsm,  // DCO control DSM input (binary-coded)
    input reset_n,	            // DCO start-up signal for current reference
    `output_xbit clk_outp,      // positive output clock
    `output_xbit clk_outn       // negative output clock
);

    parameter real PN_fcenter = freq_center;

    reg [9:0] in_delay;
    reg signed [2:0] in_dsm_delay;
    reg [30:0] int_coarse, int_fine;    // coarse & fine code (thermometer-coded)
    reg [6:0] int_dsm;                  // DSM code (thermometer-coded)
    reg [6:0] prop_gain;

    xbit [9:0] in_xbit, in_delay_xbit;
    xbit [2:0] in_dsm_xbit, in_dsm_delay_xbit;

    bit_to_xbit #(.width(10)) conn_in(.in(in), .out(in_xbit));
    bit_to_xbit #(.width(3)) conn_in_dsm(.in(in_dsm), .out(in_dsm_xbit));

    genvar j;
    generate 
        for (j=0; j<10; j=j+1) begin : gen_loop_in
        delay_xbit #(.delay(delay_in_int))
            delay_xbit_in(.in(in_xbit[j]), .out(in_delay_xbit[j]));
        end
        for (j=0; j<3; j=j+1) begin : gen_loop_in_dsm
        delay_xbit #(.delay(delay_in_int))
            delay_xbit_in_dsm(.in(in_dsm_xbit[j]), .out(in_dsm_delay_xbit[j]));
        end
    endgenerate

    xbit_to_bit #(.width(10)) conn_in_delay(.in(in_delay_xbit), .out(in_delay));
    xbit_to_bit #(.width(3)) conn_in_dsm_delay(.in(in_dsm_delay_xbit), .out(in_dsm_delay));

    // binary to thermometer encoder with retimer
    lcdco_encoder therm_encoder (
            .clk(clk_lf), .reset_n(reset_n),
            .in(in_delay), .in_dsm(in_dsm_delay), .dco_pgain(dco_pgain),
            .int_coarse(int_coarse), .int_fine(int_fine), .int_dsm(int_dsm),
            .prop_gain(prop_gain)
        );

    // core dco instances
    lcdco_core #(.num_phase(2),
        .int_coarse_steps(31), .int_fine_steps(31), .int_dsm_steps(7),
        .freq_center(freq_center), .K_dco(K_dco),
        .PN_fcenter(PN_fcenter), .PN_foffset(PN_foffset), .PN_dbc(PN_dbc),
        .delay_in_int(delay_cq), .delay_in_prop(delay_in_prop),
        .t_const_int(t_const_int), .t_const_prop(t_const_prop))
        lcdco_core (.vdd(vdd), .vss(vss), .reset_n(reset_n),
                .prop_gain(prop_gain), .prop_up(prop_up), .prop_dn(prop_dn),
                .int_coarse(int_coarse), .int_fine(int_fine), .int_dsm(int_dsm),
                .out({clk_outn,clk_outp}));

endmodule

module lcdco_core #(
    `parameter_integer(num_phase, 2),           // number of output phases
    `parameter_integer(int_coarse_steps, 31),   // integral control bits (thermometer/coarse)
    `parameter_integer(int_fine_steps, 31),     // integral control bits (thermometer/fine)
    `parameter_integer(int_dsm_steps, 7),       // integral control bits (thermometer/DSM)
    `parameter_real(K_dco, 0.00010),        // DCO relative gain
    `parameter_real(freq_center, 8.0e9),    // center of the DCO output frequency
    `parameter_real(init_phase, M_PI/2),    // initial phase (randomized if <0)
    `parameter_real(RJ_kappa, 0.0e-7),      // RMS random jitter after 1-second free-running
    `parameter_real(PN_fcenter, 8.0e9),     // phase noise center frequency
    `parameter_real(PN_foffset, 1.0e6),     // phase noise offset frequency
    `parameter_real(PN_dbc, -110),          // phase noise measured in dBc
    `parameter_real(delay_in_int, 150.0e-12),   // integral path input delay
    `parameter_real(delay_in_prop, 200.0e-12),  // proportional path input delay
    `parameter_real(t_const_int, 100.0e-12),    // integral path time-constant
    `parameter_real(t_const_prop, 40.0e-12)     // proportional path time-constant
)(
    input vdd,
    input vss,
    input reset_n,                          // reset (active-low)
    `input_xbit prop_up,                    // proportional up input pulse bit
    `input_xbit prop_dn,                    // proportional down input pulse bit
    input [int_coarse_steps-1:0] int_coarse,// integral input code (thermometer/coarse)
    input [int_fine_steps-1:0] int_fine,    // integral input code (thermometer/fine)
    input [int_dsm_steps-1:0] int_dsm,      // DSM input code (thermometer-coded)
    input [6:0] prop_gain,                  // DCO proportional gain control (thermometer-coded)
    `output_xbit [num_phase-1:0] out        // output clocks
);

    // parameters
    parameter real BW_int = 1/t_const_int;
    parameter real BW_prop = 1/t_const_prop;

    // variables
    real prop_gain_r;
    real freq;
    xreal freq_xr;

    bit [num_phase-1:0] out_init;
    xbit [num_phase-1:0] out_init_xbit;
    xbit [num_phase-1:0] out_xbit;
    xbit rstn;

    xreal int_ctrl_xr, dsm_ctrl_xr, prop_up_xr, prop_dn_xr;
    xreal int_ctrl_filter, dsm_ctrl_filter, prop_up_filter, prop_dn_filter;
    xreal int_factor, prop_factor;
    real int_factor_r, prop_factor_r;

    integer i;

    // DCO proportional gain control
    initial begin
        for(i=0; i<num_phase; i=i+1) begin
            if(((i+1)/2 - i/2) == 1) out_init[i] = 1'b1;
            else out_init[i] = 1'b0;
        end
    end

    bit_to_xbit #(.width(num_phase)) conn0(.in(out_init), .out(out_init_xbit));

    always @(prop_gain) begin
        case (prop_gain)
            7'b0000001 : prop_gain_r = 0.5*32;
            7'b0000011 : prop_gain_r = 1*32;
            7'b0000111 : prop_gain_r = 2*32;
            7'b0001111 : prop_gain_r = 4*32;
            7'b0011111 : prop_gain_r = 8*32;
            7'b0111111 : prop_gain_r = 16*32;
            7'b1111111 : prop_gain_r = 32*32;
            default : prop_gain_r = 0*32;
        endcase
    end

    real int_ctrl, dsm_ctrl;

    always @(int_coarse or int_fine or int_dsm) begin
        // integral contrl
        int_ctrl = 0.0;
        dsm_ctrl = 0.0;
        for (i = 0; i < int_coarse_steps; i=i+1)
            int_ctrl = int_ctrl+int_coarse[i];
        int_ctrl =int_ctrl*(int_coarse_steps + 1);

        for (i = 0; i < int_fine_steps; i=i+1)
            int_ctrl = int_ctrl+int_fine[i];

        for (i = 0; i < int_dsm_steps; i=i+1)
            dsm_ctrl = dsm_ctrl+int_dsm[i];
    end

    real_to_xreal conn_int(.in(int_ctrl), .out(int_ctrl_xr));
    real_to_xreal conn_dsm(.in(dsm_ctrl), .out(dsm_ctrl_xr));
    transition tran_pup(.in(prop_up), .out(prop_up_xr));
    transition tran_pdn(.in(prop_dn), .out(prop_dn_xr));

    filter #(.poles({BW_int,0.0}), .zeros({0}), .delay(delay_in_int))
            filter_int(.in(int_ctrl_xr), .out(int_ctrl_filter));

    filter #(.poles({BW_int,0.0}), .zeros({0}), .delay(delay_in_int))
            filter_dsm(.in(dsm_ctrl_xr), .out(dsm_ctrl_filter));

    filter_var #(.delay(delay_in_prop), .num_poles(1), .num_zeros(0))
            filter_prop_up(.in(prop_up_xr), .out(prop_up_filter),
                           .poles({BW_prop,0.0}), .zeros(),
                           .gain(prop_gain_r));
    filter_var #(.delay(delay_in_prop), .num_poles(1), .num_zeros(0))
            filter_prop_dn(.in(prop_dn_xr), .out(prop_dn_filter),
                           .poles({BW_prop,0.0}), .zeros(),
                           .gain(prop_gain_r));


    poly_func #(.num_in(2), .data({-(int_coarse_steps+1)*(int_fine_steps+1)/2.0,1.0,1.0})) poly_func0(.in({int_ctrl_filter,dsm_ctrl_filter}), .out(int_factor));

    add #(.scale({K_dco,-K_dco})) add0(.in({prop_up_filter,prop_dn_filter}), .out(prop_factor));

    xreal_to_real conn_int_factor(.in(int_factor), .out(int_factor_r));
    xreal_to_real conn_prop_factor(.in(prop_factor), .out(prop_factor_r));

    // digital-to-frequency conversion (exponential D-to-f)
    always @(int_factor_r or prop_factor_r) begin
        freq = freq_center * `pow(1 + K_dco, int_factor_r) * (1 + prop_factor_r);
    end

    real_to_xreal conn1(.in(freq), .out(freq_xr));

    // frequency-to-clock conversion
    freq_to_clk #(.init_phase(init_phase), .num_phase(num_phase),
                .RJ_kappa(RJ_kappa), .PN_foffset(PN_foffset),
                .PN_fcenter(PN_fcenter), .PN_dbc(PN_dbc))
            freq2clk(.in(freq_xr), .out(out_xbit));

    bit_to_xbit conn2(.in(reset_n), .out(rstn));

    genvar j;
    generate
        for (j=0; j<num_phase; j=j+1) begin : gen_loop
        mux_xbit out_gen(.in({out_xbit[j],out_init_xbit[j]}), .sel(rstn), .out(out[j]));
        end
    endgenerate

endmodule

module lcdco_encoder(
    input clk,                  // input retiming clock
    input reset_n,              // reset (active-low)
    input [9:0] in,             // binary-coded integral path input
    input signed [2:0] in_dsm,  // binary-coded DSM input (signed)
    input [2:0] dco_pgain,      // bianry-coded proportional path gain control code
    output reg [30:0] int_coarse,   // thermometer-coded integral code (coarse)
    output reg [30:0] int_fine,     // thermometer-coded integral code (fine)
    output reg [6:0] int_dsm,       // thermometer-coded integral code (DSM)
    output reg [6:0] prop_gain      // thermometer-ocded proportional path gain control code
);

    reg [9:0] in_retimed;
    reg [2:0] in_dsm_retimed;
    reg [2:0] dco_pgain_retimed;

    // binary to thermometer decoder
    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            in_retimed <= 10'b10_0000_0000;
            in_dsm_retimed <= 3'b000;
            dco_pgain_retimed <= 3'b000;
        end
        else begin
            in_retimed <= in;
            in_dsm_retimed <= in_dsm;
            dco_pgain_retimed <= dco_pgain;
        end
    end

    lcdco_therm_encoder therm_coarse(.din(in_retimed[9:5]), .dout(int_coarse));
    lcdco_therm_encoder therm_fine(.din(in_retimed[4:0]), .dout(int_fine));

    lcdco_therm_encoder_dsm therm_dsm(.din(in_dsm_retimed), .dout(int_dsm));
    lcdco_therm_encoder_pgain therm_pgain(.din(dco_pgain_retimed), .dout(prop_gain));

endmodule

module lcdco_therm_encoder( 
    input [4:0] din,        // binary-coded input
    output reg [30:0] dout  // thermometer-coded output
);

    // binary-to-thermometer encoder
    always @(din) begin
        case(din)
            5'b00000 : dout = 31'b0000000000000000000000000000000;
            5'b00001 : dout = 31'b0000000000000000000000000000001;
            5'b00010 : dout = 31'b0000000000000000000000000000011;
            5'b00011 : dout = 31'b0000000000000000000000000000111;
            5'b00100 : dout = 31'b0000000000000000000000000001111;
            5'b00101 : dout = 31'b0000000000000000000000000011111;
            5'b00110 : dout = 31'b0000000000000000000000000111111;
            5'b00111 : dout = 31'b0000000000000000000000001111111;

            5'b01000 : dout = 31'b0000000000000000000000011111111;
            5'b01001 : dout = 31'b0000000000000000000000111111111;
            5'b01010 : dout = 31'b0000000000000000000001111111111;
            5'b01011 : dout = 31'b0000000000000000000011111111111;
            5'b01100 : dout = 31'b0000000000000000000111111111111;
            5'b01101 : dout = 31'b0000000000000000001111111111111;
            5'b01110 : dout = 31'b0000000000000000011111111111111;
            5'b01111 : dout = 31'b0000000000000000111111111111111;

            5'b10000 : dout = 31'b0000000000000001111111111111111;
            5'b10001 : dout = 31'b0000000000000011111111111111111;
            5'b10010 : dout = 31'b0000000000000111111111111111111;
            5'b10011 : dout = 31'b0000000000001111111111111111111;
            5'b10100 : dout = 31'b0000000000011111111111111111111;
            5'b10101 : dout = 31'b0000000000111111111111111111111;
            5'b10110 : dout = 31'b0000000001111111111111111111111;
            5'b10111 : dout = 31'b0000000011111111111111111111111;

            5'b11000 : dout = 31'b0000000111111111111111111111111;
            5'b11001 : dout = 31'b0000001111111111111111111111111;
            5'b11010 : dout = 31'b0000011111111111111111111111111;
            5'b11011 : dout = 31'b0000111111111111111111111111111;
            5'b11100 : dout = 31'b0001111111111111111111111111111;
            5'b11101 : dout = 31'b0011111111111111111111111111111;
            5'b11110 : dout = 31'b0111111111111111111111111111111;
            5'b11111 : dout = 31'b1111111111111111111111111111111;
        endcase
    end

endmodule

module lcdco_therm_encoder_dsm( 
    input [2:0] din,        // binary-coded input
    output reg [6:0] dout   // thermometer-coded output
);

    // binary-to-thermometer encoder
    always @(din) begin
        case(din)
            3'b100 : dout = 7'b0000000;
            3'b101 : dout = 7'b0000001;
            3'b110 : dout = 7'b0000011;
            3'b111 : dout = 7'b0000111;
            3'b000 : dout = 7'b0001111;
            3'b001 : dout = 7'b0011111;
            3'b010 : dout = 7'b0111111;
            3'b011 : dout = 7'b1111111;
        endcase
    end

endmodule

module lcdco_therm_encoder_pgain( 
    input [2:0] din,        // binary-coded input
    output reg [6:0] dout   // thermometer-coded output
);

    // binary-to-thermometer encoder
    always @(din) begin
        case(din)
            3'b000 : dout = 7'b0000000;
            3'b001 : dout = 7'b0000001;
            3'b010 : dout = 7'b0000011;
            3'b011 : dout = 7'b0000111;
            3'b100 : dout = 7'b0001111;
            3'b101 : dout = 7'b0011111;
            3'b110 : dout = 7'b0111111;
            3'b111 : dout = 7'b1111111;
        endcase
    end

endmodule
/*----------------------------------------------------------------------
MODULE gro_tdc.sv

= Purpose =

A gated relaxation oscillator type time-to-digital converter.

= Description =

The gated relaxation oscillator type time-to-digital converter block.

= Revisions =

$Author$
$DateTIme$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module gro_tdc #(
    `parameter_real(K_tdc, 64),             // TDC gain
    `parameter_real(delay_rst, 131e-12),    // PFD reset propagation delay
    `parameter_real(delay_cq, 315e-12),     // clock-to-output delay
    `parameter_real(delay_cp, 202e-12),     // clock-to-proportional path output delay
    `parameter_realarray(f_range_cof_0),       // frequency tuning coefficients for f_range=0
    `parameter_realarray(f_range_cof_1),     // frequency tuning coefficients for f_range=1
    `parameter_realarray(f_range_cof_2),   // frequency tuning coefficients for f_range=2
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 2.1e4),        // equivalent output resistance
    `parameter_real(res_out_prop, 0.26e4)   // equivalent output resistance (proportional term)
)(
    input vdd,
    input vss,
    input reset_n,          // reset (active low)
    input [1:0] f_range,    // frequency range
    `input_xbit clk_ref,    // reference clock input
    `input_xbit clk_fb,     // feedback clock input
    output reg [5:0] up,    // timing error output (up signal)
    output reg [5:0] dn,    // timing error output (dn signal)
    `output_xbit prop_up,   // PFD output pulse (up signal)
    `output_xbit prop_dn    // PFD output pulse (dn signal)
);

    // parameters
    parameter real prop_up_delay = delay_cp;
    parameter real prop_dn_delay = delay_cp;

    //variables
    wire pulse;
    reg clk_fdc;
    reg [5:0] f_bias, fdc_ctrl;

    xbit_to_bit conn0(.in(clk_ref), .out(clk_fdc));

    // GRO_TDC core
    gro_tdc_core #(.delay_rst(delay_rst),
                   .delay_cq(delay_cq),
                   .prop_up_delay(prop_up_delay),
                   .prop_dn_delay(prop_dn_delay),
                   .K_tdc(K_tdc),
                   .f_range_cof_0(f_range_cof_0),
                   .f_range_cof_1(f_range_cof_1),
                   .f_range_cof_2(f_range_cof_2))
            gro_tdc_core (
                .vdd(vdd), .vss(vss),
                .reset_n(reset_n), .f_range(f_range),
                .f_bias(f_bias), .fdc_ctrl(fdc_ctrl),
                .clk_ref(clk_ref), .clk_fb(clk_fb),
                .up(up), .dn(dn), .prop_up(prop_up), .prop_dn(prop_dn),
                .pulse(pulse));

    // FDC logic
    tdc_fdc_logic tdc_fdc_logic (
                    .clk(clk_fdc), .reset_n(reset_n), .pulse(pulse),
                    .dctrl(fdc_ctrl), .out(f_bias));

endmodule

module gray_counter (
    `input_xbit clk,
    input reset_n,
    output reg [4:0] gray_out
);
    reg [4:0] gray_in;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            gray_in <= 5'b0;
        end
        else begin
            gray_in <= gray_in + 1;
        end
    end

    always @(gray_in) begin
        case(gray_in)
            5'b00000 : gray_out = 5'b00000;
            5'b00001 : gray_out = 5'b00001;
            5'b00010 : gray_out = 5'b00011;
            5'b00011 : gray_out = 5'b00010;
            5'b00100 : gray_out = 5'b00110;
            5'b00101 : gray_out = 5'b00111;
            5'b00110 : gray_out = 5'b00101;
            5'b00111 : gray_out = 5'b00100;
                                     
            5'b01000 : gray_out = 5'b01100;
            5'b01001 : gray_out = 5'b01101;
            5'b01010 : gray_out = 5'b01111;
            5'b01011 : gray_out = 5'b01110;
            5'b01100 : gray_out = 5'b01010;
            5'b01101 : gray_out = 5'b01011;
            5'b01110 : gray_out = 5'b01001;
            5'b01111 : gray_out = 5'b01000;
                                     
            5'b10000 : gray_out = 5'b11000;
            5'b10001 : gray_out = 5'b11001;
            5'b10010 : gray_out = 5'b11011;
            5'b10011 : gray_out = 5'b11010;
            5'b10100 : gray_out = 5'b11110;
            5'b10101 : gray_out = 5'b11111;
            5'b10110 : gray_out = 5'b11101;
            5'b10111 : gray_out = 5'b11100;
                                     
            5'b11000 : gray_out = 5'b10100;
            5'b11001 : gray_out = 5'b10101;
            5'b11010 : gray_out = 5'b10111;
            5'b11011 : gray_out = 5'b10110;
            5'b11100 : gray_out = 5'b10010;
            5'b11101 : gray_out = 5'b10011;
            5'b11110 : gray_out = 5'b10001;
            5'b11111 : gray_out = 5'b10000;
            default  : gray_out = 5'b00000;
        endcase
    end

endmodule

module gray_to_bin (
    input [4:0] in,
    output reg [4:0] out
);

    always @(in) begin
        case(in)
            5'b00000 : out = 5'b00000;
            5'b00001 : out = 5'b00001;
            5'b00011 : out = 5'b00010;
            5'b00010 : out = 5'b00011;
            5'b00110 : out = 5'b00100;
            5'b00111 : out = 5'b00101;
            5'b00101 : out = 5'b00110;
            5'b00100 : out = 5'b00111;

            5'b01100 : out = 5'b01000;
            5'b01101 : out = 5'b01001;
            5'b01111 : out = 5'b01010;
            5'b01110 : out = 5'b01011;
            5'b01010 : out = 5'b01100;
            5'b01011 : out = 5'b01101;
            5'b01001 : out = 5'b01110;
            5'b01000 : out = 5'b01111;

            5'b11000 : out = 5'b10000;
            5'b11001 : out = 5'b10001;
            5'b11011 : out = 5'b10010;
            5'b11010 : out = 5'b10011;
            5'b11110 : out = 5'b10100;
            5'b11111 : out = 5'b10101;
            5'b11101 : out = 5'b10110;
            5'b11100 : out = 5'b10111;

            5'b10100 : out = 5'b11000;
            5'b10101 : out = 5'b11001;
            5'b10111 : out = 5'b11010;
            5'b10110 : out = 5'b11011;
            5'b10010 : out = 5'b11100;
            5'b10011 : out = 5'b11101;
            5'b10001 : out = 5'b11110;
            5'b10000 : out = 5'b11111;
            default  : out = 5'b00000;
        endcase
    end

endmodule

module gro #(
    `parameter_real(K_tdc, 64),                 // TDC resolution
    `parameter_realarray(f_range_cof_0), // frequency tuning coefficients for f_range=0
    `parameter_realarray(f_range_cof_1), // frequency tuning coefficients for f_range=1
    `parameter_realarray(f_range_cof_2)  // frequency tuning coefficients for f_range=2
)(
    input vdd,
    input vss,
    `input_xbit pw_in,          // input pulse width
    input bit [1:0] f_range,    // frequency range
    input bit [5:0] fdc_in,     // code converted from frequency
    input reset_n,              // reset (active low)
    `output_xbit outp,          // output clock whose period is proportional to the input pulse width (positive)
    `output_xbit outn           // output clock whose period is proportional to the input pulse width (negative)

);

    parameter real scale_factor_2 = 14;

    integer K_tdc_integer;

    real freq;
    real freq_ktdc;
    xreal freq_ktdc_xr;
    xreal freq_gro;
    integer count;

    initial begin
        K_tdc_integer = K_tdc;
        count = 0;
    end

    always @(*) begin
        if($realtime != 0) begin
            if(!reset_n) begin
                freq = 0;
            end
            else begin
                case(f_range)
                    2'b00 : freq = `pow(f_range_cof_0[0] + f_range_cof_0[1] * fdc_in + f_range_cof_0[2] * fdc_in * fdc_in, 0.5) * 1e6 / scale_factor_2;
                    2'b01 : freq = `pow(f_range_cof_1[0] + f_range_cof_1[1] * fdc_in + f_range_cof_1[2] * fdc_in * fdc_in, 0.5) * 1e6 / scale_factor_2;
                    2'b10 : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1] * fdc_in + f_range_cof_2[2] * fdc_in * fdc_in, 0.5) * 1e6 / scale_factor_2;
                    default : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1] * fdc_in + f_range_cof_2[2] * fdc_in * fdc_in, 0.5) * 1e6 / scale_factor_2;
                endcase
            end
        end
        else begin
            case(f_range)
                2'b00 : freq = `pow(f_range_cof_0[0] + f_range_cof_0[1]*63.0 + f_range_cof_0[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                2'b01 : freq = `pow(f_range_cof_1[0] + f_range_cof_1[1]*63.0 + f_range_cof_1[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                2'b10 : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                default : freq = `pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;

            endcase
        end
    end

    always @(*) begin
        freq_ktdc = freq * K_tdc_integer / 2.0;
    end

    real_to_xreal conn0(.in(freq_ktdc), .out(freq_ktdc_xr));

    select select_freq(.sel(pw_in), .in({freq_ktdc_xr,`zero_xreal}), .out(freq_gro));

    freq_to_clk #(.num_phase(2), .init_phase(0.0))
            freq2clk(.in(freq_gro), .out({outp,outn}));

endmodule

module gro_tdc_core #(
    `parameter_real(K_tdc, 64),                 // TDC gain
    `parameter_real(delay_rst, 10.0e-12),       // PFD reset propagation delay
    `parameter_real(delay_cq, 100.0e-12),       // clock-to-output delay
    `parameter_real(prop_up_delay, 150.0e-12),  // clock-to-proportional path output (up) delay
    `parameter_real(prop_dn_delay, 150.0e-12),  // clock-to-proportional path output (dn) delay
    `parameter_realarray(f_range_cof_0), // frequency tuning coefficients for f_range=0
    `parameter_realarray(f_range_cof_1), // frequency tuning coefficients for f_range=1
    `parameter_realarray(f_range_cof_2)  // frequency tuning coefficients for f_range=2
)(
    input vdd,
    input vss,
    input reset_n,          // reset (active low)
    input [1:0] f_range,    // frequency range
    input [5:0] f_bias,     // frequency calibration code
    input [5:0] fdc_ctrl,   // control code of relaxation osc for FDC
    `input_xbit clk_ref,    // reference clock input
    `input_xbit clk_fb,     // feedback clock input
    output reg [5:0] up,    // timing error output (up signal)
    output reg [5:0] dn,    // timing error output (dn signal)
    `output_xbit prop_up,   // PFD output pulse (up signal)
    `output_xbit prop_dn,   // PFD output pulse (dn signal)
    output reg pulse        // pulse signal to operate fdc_logic
);

    xbit up_pfd, dn_pfd;

    xbit clk_up_p, clk_up_n, clk_dn_p, clk_dn_n;

    reg [4:0] count_up0_gray, count_up1_gray, count_dn0_gray, count_dn1_gray;
    reg [4:0] up_sample0, up_sample1, dn_sample0, dn_sample1;
    reg [4:0] up_sample0_gray, up_sample1_gray, dn_sample0_gray, dn_sample1_gray;
    reg [4:0] up_prev0, up_prev1, dn_prev0, dn_prev1;
    reg [4:0] up0, up1, dn0, dn1;
    reg [5:0] up_sum, dn_sum;

    gro_tdc_pfd #(.delay_rst(delay_rst))
        pfd(.up(up_pfd), .dn(dn_pfd),
        .clk_ref(clk_ref), .clk_fb(clk_fb));

    gro #(.K_tdc(K_tdc),
          .f_range_cof_0(f_range_cof_0),
          .f_range_cof_1(f_range_cof_1),
          .f_range_cof_2(f_range_cof_2))
        gro_up(.vdd(vdd), .vss(vss),
            .pw_in(up_pfd), .f_range(f_range), .fdc_in(f_bias),
            .reset_n(reset_n), .outp(clk_up_p), .outn(clk_up_n));
    gro #(.K_tdc(K_tdc),
          .f_range_cof_0(f_range_cof_0),
          .f_range_cof_1(f_range_cof_1),
          .f_range_cof_2(f_range_cof_2))
        gro_dn(.vdd(vdd), .vss(vss),
            .pw_in(dn_pfd), .f_range(f_range), .fdc_in(f_bias),
            .reset_n(reset_n), .outp(clk_dn_p), .outn(clk_dn_n));
    
    gray_counter gray_up0(.clk(clk_up_p), .reset_n(reset_n), .gray_out(count_up0_gray));
    gray_counter gray_up1(.clk(clk_up_n), .reset_n(reset_n), .gray_out(count_up1_gray));
    gray_counter gray_dn0(.clk(clk_dn_p), .reset_n(reset_n), .gray_out(count_dn0_gray));
    gray_counter gray_dn1(.clk(clk_dn_n), .reset_n(reset_n), .gray_out(count_dn1_gray));

    gray_to_bin bin_up0(.in(up_sample0_gray), .out(up_sample0));
    gray_to_bin bin_up1(.in(up_sample1_gray), .out(up_sample1));
    gray_to_bin bin_dn0(.in(dn_sample0_gray), .out(dn_sample0));
    gray_to_bin bin_dn1(.in(dn_sample1_gray), .out(dn_sample1));

    assign up0 = up_sample0 + ~up_prev0 + 5'b00001;
    assign up1 = up_sample1 + ~up_prev1 + 5'b00001;
    assign dn0 = dn_sample0 + ~dn_prev0 + 5'b00001;
    assign dn1 = dn_sample1 + ~dn_prev1 + 5'b00001;
    assign up_sum = {1'b0,up0} + {1'b0,up1};
    assign dn_sum = {1'b0,dn0} + {1'b0,dn1};

    // counted value sampling (timing error between two adjacent negative edges of the feedback clock)
    always @(negedge clk_fb or negedge reset_n) begin
        if(!reset_n) begin
            up_prev0 <= 5'b0;
            up_prev1 <= 5'b0;
            up_sample0_gray <= 5'b0;
            up_sample1_gray <= 5'b0;
        end
        else begin
            up_prev0 <= up_sample0;
            up_prev1 <= up_sample1;
            up_sample0_gray <= count_up0_gray;
            up_sample1_gray <= count_up1_gray;
        end
    end

    always @(negedge clk_fb or negedge reset_n) begin
        if(!reset_n) begin
            dn_prev0 <= 5'b0;
            dn_prev1 <= 5'b0;
            dn_sample0_gray <= 5'b0;
            dn_sample1_gray <= 5'b0;
        end
        else begin
            dn_prev0 <= dn_sample0;
            dn_prev1 <= dn_sample1;
            dn_sample0_gray <= count_dn0_gray;
            dn_sample1_gray <= count_dn1_gray;
        end
    end

    // output retiming
    always @(posedge clk_fb or negedge reset_n) begin
        if(!reset_n) begin
            up <= 6'b0;
            dn <= 6'b0;
        end
        else begin
            up <= #(delay_cq/`TIME_SCALE) up_sum;
            dn <= #(delay_cq/`TIME_SCALE) dn_sum;
        end
    end

    // PFD output pulse
    delay_xbit #(.delay(prop_up_delay)) up_delay(.in(up_pfd), .out(prop_up));
    delay_xbit #(.delay(prop_dn_delay)) dn_delay(.in(dn_pfd), .out(prop_dn));

    // relaxation oscillator and pulse generator for the frequency to digital converter (frequency auto calibration for constant Ktdc)
    tdc_relaxation_osc #(.scale_factor(K_tdc/2),
                         .f_range_cof_0(f_range_cof_0),
                         .f_range_cof_1(f_range_cof_1),
                         .f_range_cof_2(f_range_cof_2))
            tdc_relaxation_osc(.vdd(vdd), .vss(vss),
                    .f_range(f_range), .fdc_in(fdc_ctrl),
                    .reset_n(reset_n), .clk_relax_osc(clk_osc));

    xbit_to_bit conn(.in(clk_ref), .out(clk_fdc));

    tdc_fdc_controller #(.K_tdc(K_tdc))
            tdc_fdc_controller(.vdd(vdd), .vss(vss), .clk_in(clk_fdc),
                    .clk_osc(clk_osc), .reset_n(reset_n), .pulse(pulse));

endmodule

module gro_tdc_pfd #(
    `parameter_real(delay_rst, 0.0)     // reset propagation delay
)(
    `output_xbit up,	                // up signal
    `output_xbit dn,	                // down signal
    `input_xbit clk_ref,                // reference clock
    `input_xbit clk_fb                  // feedback clock
);

    // variables
    xbit rst;

    // two dffs with asynchronous reset
    dff_rst_async_xbit #(.init_value(0)) dff_up(.q(up), .d(`one_xbit), .clk(clk_ref), .rst(rst));
    dff_rst_async_xbit #(.init_value(0)) dff_dn(.q(dn), .d(`one_xbit), .clk(clk_fb), .rst(rst));

    // reset path
    and2_xbit #(.delay(delay_rst)) and_xbit(.out(rst),.in_a(up),.in_b(dn));

endmodule

module tdc_fdc_acc (
    input clk,
    input reset_n,
    input rst_sync_global,
    input rst_sync_local,
    input enable,
    input [4:0] in,
    output reg [3:0] n_up,
    output reg [3:0] n_dn
);

    reg [3:0] n_up_nxt, n_dn_nxt;
    reg [1:0] flag;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) flag <= 2'b0;
        else begin
            if(rst_sync_local) begin
                flag <= 2'b0;
            end
            else begin
                if(in == 5'b01110) begin     // 15 - 1
                    if(flag[1]) begin
                        flag <= 2'b11;
                    end
                    else begin
                        flag <= 2'b01;
                    end
                end
                else begin
                    if(in == 5'b01111) begin // 15+1 - 1
                        flag <= 2'b11;
                    end
                    else begin
                        flag <= flag;
                    end
                end
            end
        end
    end
    
    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            n_up <= 4'b0;
            n_dn <= 4'b0;
        end
        else begin
            if(rst_sync_global) begin
                n_up <= 4'b0;
                n_dn <= 4'b0;
            end
            else begin
                if(enable) begin
                    n_up <= n_up_nxt;
                    n_dn <= n_dn_nxt;
                end
                else begin
                    n_up <= n_up;
                    n_dn <= n_dn;
                end
            end
        end
    end

    always @(*) begin
        case(flag)
            2'b00 :
                begin
                    n_up_nxt = n_up;
                    n_dn_nxt = n_dn + 1;
                end
            2'b11 :
                begin
                    n_up_nxt = n_up + 1;
                    n_dn_nxt = n_dn;
                end
            default :
                begin
                    n_up_nxt = n_up;
                    n_dn_nxt = n_dn;
                end
        endcase
    end

endmodule

module tdc_fdc_controller #(
    `parameter_real(K_tdc, 64)
)(
    input vdd,
    input vss,
    input clk_in,               // clock whose frequency is going to be acquired
    input clk_osc,              // oscillator operating frequency
    input reset_n,              // reset (active low)
    output reg pulse            // pulse signal to operate fdc_logic
);

    reg clk_osc_div_ktdc, clk_osc_div;

    tdc_fdc_div #(.div_factor(K_tdc/2))
            fdc_div_ktdc(.clk(clk_osc), .reset_n(reset_n),
                         .clk_div(clk_osc_div_ktdc));

    tdc_fdc_div #(.div_factor(16))
            fdc_div(.clk(clk_osc_div_ktdc), .reset_n(reset_n),
                    .clk_div(clk_osc_div));

    tdc_fdc_pulse_gen fdc_pulse_gen(.clk(clk_in), .in(clk_osc_div), .reset_n(reset_n), .pulse(pulse));


endmodule

module tdc_fdc_counter (
    input clk,
    input reset_n,
    input rst_sync,
    output reg [4:0] cnt
);

    reg [4:0] cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 5'b0;
        end
        else begin
            if(rst_sync) begin
                cnt <= 5'b0;
            end
            else begin
                cnt <= cnt_nxt;
            end
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module tdc_fdc_counter_en (
    input clk,
    input reset_n,
    input enable,
    output reg [3:0] cnt
);

    reg [3:0] cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 4'b0;
        end
        else begin
            if(enable)
                if(cnt == 4'b1110) cnt <= 4'b0;
                else cnt <= cnt_nxt;
            else
                cnt <= cnt;
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module tdc_fdc_decision (
    input clk,
    input reset_n,
    input enable,
    input enable_global,
    input [3:0] n_up,
    input [3:0] n_dn,
    output reg [5:0] interior_code,
    output reg [5:0] out
);

    wire change;
    reg n_up_msb_pre, n_dn_msb_pre;
    reg [5:0] interior_code_nxt, exterior_code_nxt, history;

    assign change = ((n_up[3]^n_up_msb_pre) | (n_dn[3]^n_dn_msb_pre)) ? 1'b1 : 1'b0;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            interior_code <= 6'b100000; //6'b111111;
            out <= 6'b100000; //6'b111111;
        end
        else begin
            out <= exterior_code_nxt;
            if(enable) interior_code <= interior_code_nxt;
            else interior_code <= interior_code;
        end
    end

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            n_up_msb_pre <= 1'b0;
            n_dn_msb_pre <= 1'b0;
        end
        else begin
            if(enable_global) begin
                n_up_msb_pre <= n_up[3];
                n_dn_msb_pre <= n_dn[3];
            end
            else begin
                n_up_msb_pre <= n_up_msb_pre;
                n_dn_msb_pre <= n_dn_msb_pre;
            end
        end
    end

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            history <= 6'b0;
        end
        else begin
            if(enable_global) begin
                if(change) history <= {history[5:0],1'b1};
                else history <= {history[5:0],1'b0};
            end
            else begin
                history <= history;
            end
        end
    end

    always @(*) begin
        if(n_up[3]) begin
            if(interior_code == 6'b111111) interior_code_nxt = 6'b111111;
            else interior_code_nxt = interior_code + 1;
        end
        else if(n_dn[3]) begin
            if(interior_code == 6'b000000) interior_code_nxt = 6'b000000;
            else interior_code_nxt = interior_code - 1;
        end
        else begin
            interior_code_nxt = interior_code;
        end
    end

    always @(*) begin
        if(&history) exterior_code_nxt = out;
        else exterior_code_nxt = interior_code;
    end

endmodule

module tdc_fdc_div #(
    `parameter_integer(div_factor, 32)
)(
    input clk,
    input reset_n,
    output reg clk_div
);

    reg [9:0] cnt, cnt_nxt;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            cnt <= 0;
            clk_div <= 0;
        end
        else begin
            if(cnt == div_factor/2-1) begin
                cnt <= 0;
                clk_div <= ~clk_div;
            end
            else begin
                cnt <= cnt_nxt;
            end
        end
    end

    always @(cnt) begin
        cnt_nxt=cnt+1;
    end

endmodule

module tdc_fdc_logic (
    input clk,                  // clock whose frequency is going to be acquired
    input reset_n,              // reset (active low)
    input pulse,                // pulse detecting the rising edge of the divided clk_osc
    output reg [5:0] dctrl,     // control code only in FDC
    output reg [5:0] out        // control code output
);

    wire en_cnt, en_acc, en_dec, en_dec_global, rst_cnt, rst_acc_global, rst_acc_local;
    reg period, period_retimed, pulse_acc, pulse_retimed;
    reg [4:0] cnt;
    reg [3:0] n_up, n_dn, cnt_period;

    assign en_cnt = pulse;
    assign en_acc = pulse;
    assign en_dec = period;
    assign en_dec_global = pulse_acc;
    assign rst_cnt = pulse;
    assign rst_acc_global = pulse_acc;
    assign rst_acc_local = pulse_retimed;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            pulse_retimed <= 1'b0;
            period_retimed <= 1'b0;
        end
        else begin
            pulse_retimed <= pulse;
            period_retimed <= period;
        end
    end

    always @(*) begin
        pulse_acc = period&(~period_retimed);
        if(cnt_period == 4'b1110) period = 1'b1;
        else period = 1'b0;
    end 

    tdc_fdc_counter fdc_counter_fd(.clk(clk), .reset_n(reset_n), .rst_sync(rst_cnt), .cnt(cnt));

    tdc_fdc_counter_en fdc_counter_acc(.clk(clk), .reset_n(reset_n), .enable(en_cnt), .cnt(cnt_period));

    tdc_fdc_acc fdc_acc(.clk(clk), .reset_n(reset_n), .rst_sync_global(rst_acc_global), .rst_sync_local(rst_acc_local), .enable(en_acc), .in(cnt), .n_up(n_up), .n_dn(n_dn));

    tdc_fdc_decision fdc_decision(.clk(clk), .reset_n(reset_n), .enable(en_dec), .enable_global(en_dec_global), .n_up(n_up), .n_dn(n_dn), .interior_code(dctrl), .out(out));

endmodule

module tdc_fdc_pulse_gen (
    input clk,
    input in,
    input reset_n,
    output reg pulse
);

    reg reg1, reg2;

    always @(posedge clk or negedge reset_n) begin
        if(!reset_n) begin
            reg1 <= 1'b0;
            reg2 <= 1'b0;
        end
        else begin
            reg1 <= in;
            reg2 <= reg1;
        end
    end

    always @(*) begin
        pulse = reg1&(~reg2);
    end

endmodule

module tdc_relaxation_osc #(
    `parameter_real(scale_factor, 32),          // frequency multiplication
    `parameter_realarray(f_range_cof_0), // frequency tuning coefficients for f_range=0
    `parameter_realarray(f_range_cof_1), // frequency tuning coefficients for f_range=1
    `parameter_realarray(f_range_cof_2)  // frequency tuning coefficients for f_range=2
)(
    input vdd,
    input vss,
    input [1:0] f_range,        // operation frequency range 
    input [5:0] fdc_in,         // code converted from frequency
    input reset_n,              // reset (active low)
    output clk_relax_osc        // oscillation frequnecy
);

    parameter real scale_factor_2 = 14;

    real freq;
    xreal freq_xr;
    xbit clk_relax_osc_xbit;

    always @(*) begin
        if($realtime != 0) begin
            if(!reset_n) begin
                freq = 0;
            end
            else begin
                case(f_range)
                    2'b00 : freq = scale_factor * `pow(f_range_cof_0[0] + f_range_cof_0[1] * fdc_in + f_range_cof_0[2] * fdc_in * fdc_in, 0.5) * 1e6 / scale_factor_2;
                    2'b01 : freq = scale_factor * `pow(f_range_cof_1[0] + f_range_cof_1[1] * fdc_in + f_range_cof_1[2] * fdc_in * fdc_in, 0.5) * 1e6 / scale_factor_2;
                    2'b10 : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1] * fdc_in + f_range_cof_2[2] * fdc_in * fdc_in, 0.5) * 1e6 / scale_factor_2;
                    default : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1] * fdc_in + f_range_cof_2[2] * fdc_in * fdc_in, 0.5) * 1e6 / scale_factor_2;
                endcase
            end
        end
        else begin
            case(f_range)
                2'b00 : freq = scale_factor * `pow(f_range_cof_0[0] + f_range_cof_0[1]*63.0 + f_range_cof_0[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                2'b01 : freq = scale_factor * `pow(f_range_cof_1[0] + f_range_cof_1[1]*63.0 + f_range_cof_1[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                2'b10 : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
                default : freq = scale_factor * `pow(f_range_cof_2[0] + f_range_cof_2[1]*63.0 + f_range_cof_2[2]*63.0*63.0, 0.5) * 1e6 / scale_factor_2;
            endcase
        end
    end

    real_to_xreal conn0(.in(freq), .out(freq_xr));
    freq_to_clk freq2clk(.in(freq_xr), .out(clk_relax_osc_xbit));
    xbit_to_bit conn1(.in(clk_relax_osc_xbit), .out(clk_relax_osc));

endmodule
/*----------------------------------------------------------------------
MODEL bbpfd.sv

= Purpose =

A bang-bang phase-frequency detector model to compare the phase or frequency differences between the two input clocks, providing the polarity information only.

= Description =

The bang-bang phase-frequency detector measures the phase or frequency differences between the two input clocks and expresses its polarity in binary outputs (up and down).

= Revisions =

$Author$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module bbpfd #(
    `parameter_real(delay_rst, 131e-12),    // reset propagation delay
    `parameter_real(delay_cq, 311e-12),     // clock-to-output delay
    `parameter_real(delay_cp, 199e-12),     // clock-to-proportional path output delay
    `parameter_real(cap_in, 1.8e-15),       // equivalent input capacitance
    `parameter_real(res_out, 2.1e4),        // equivalent output resistance
    `parameter_real(res_out_prop, 0.26e4)   // equivalent output resistance (proportional term)
)(
    input vdd,
    input vss,
    `input_xbit clk_ref,    // reference clock input
    `input_xbit clk_fb,     // feedback clock input
    output up,              // output (up)
    output dn,              // output (down)
    `output_xbit prop_up,   // PFD output pulse (up)
    `output_xbit prop_dn    // PFD output pusle (dn)
);

    // parameters
    parameter real prop_up_delay = delay_cp;
    parameter real prop_dn_delay = delay_cp;

    // variables
    xbit up_l, dn_l, up_m, dn_m, upb_m, dnb_m, up_n, dn_n;
    xbit up_pre_xbit, dn_pre_xbit, up_xbit, dn_xbit;

    // phase-frequency detector
    bbpfd_pfd #(.delay_rst(delay_rst))
        pfd(.clk_ref(clk_ref), .clk_fb(clk_fb),
            .up(up_l), .dn(dn_l));

    // connectors
    xbit_to_bit xbit2bit0(.in(up_xbit), .out(up));
    xbit_to_bit xbit2bit1(.in(dn_xbit), .out(dn));

    // S'R'-NAND-latch : preceding edge detection
    nand_xbit latch_00(.in({up_l, dn_m}), .out(up_m));
    nand_xbit latch_01(.in({dn_l, up_m}), .out(dn_m));

    // gated-S'R'-NAND-latch : edge discrimination & retiming
    inv_xbit #(.delay(15e-12)) inv0(.in(up_m), .out(upb_m));
    inv_xbit #(.delay(15e-12)) inv1(.in(dn_m), .out(dnb_m));
    nand_xbit clk_gate0(.in({upb_m,clk_fb}), .out(up_n));
    nand_xbit clk_gate1(.in({dnb_m,clk_fb}), .out(dn_n));

    nand_xbit latch_10(.in({up_n,dn_pre_xbit}), .out(up_pre_xbit));
    nand_xbit latch_11(.in({dn_n,up_pre_xbit}), .out(dn_pre_xbit));
    
    delay_xbit #(.delay(delay_cq)) up_delay0(.in(up_pre_xbit), .out(up_xbit));
    delay_xbit #(.delay(delay_cq)) dn_delay0(.in(dn_pre_xbit), .out(dn_xbit));

    // PFD output pulse
    delay_xbit #(.delay(prop_up_delay)) up_delay(.in(up_l), .out(prop_up));
    delay_xbit #(.delay(prop_dn_delay)) dn_delay(.in(dn_l), .out(prop_dn));

endmodule

module bbpfd_pfd #(
    `parameter_real(delay_rst, 0.0)     // reset propagation delay
)(
    `output_xbit up,	                // up signal
    `output_xbit dn,	                // down signal
    `input_xbit clk_ref,                // reference clock
    `input_xbit clk_fb                  // feedback clock
);

    // variables
    xbit rst;

    // two dffs with asynchronous reset
    dff_rst_async_xbit #(.init_value(0)) dff_up(.q(up), .d(`one_xbit), .clk(clk_ref), .rst(rst));
    dff_rst_async_xbit #(.init_value(0)) dff_dn(.q(dn), .d(`one_xbit), .clk(clk_fb), .rst(rst));

    // reset path
    and2_xbit #(.delay(delay_rst)) and_xbit(.out(rst),.in_a(up),.in_b(dn));

endmodule
/*----------------------------------------------------------------------
MODEL bbpd.sv

= Purpose =

A bang-bang phase detector model to compare the phase difference between the two input clocks, providing the polarity information only.

= Description =

The bang-bang phase-frequency detector measures the phase differences between the two input clocks and expresses its polarity in binary outputs (up and down).

= Revisions =

$Author$
$DateTime$
$Id$
----------------------------------------------------------------------*/

`include "xmodel.h"

module bbpd #(
    `parameter_real(t_offset, 20e-12),          // zero-output timing offset
    `parameter_real(clk_to_q_delay, 189e-12),   // clock-to-output delay
    `parameter_real(cap_in, 1.8e-15),           // equivalent input capacitance
    `parameter_real(res_out, 2.1e4)             // equivalent output resistance
)(
    input vdd,
    input vss,
    `input_xbit clk_ref,    // reference clock input
    `input_xbit clk_fb,     // feedback clock input
    output up,              // output (up)
    output dn               // output (down)
);

    parameter real delay_ref = (t_offset > 0) ? t_offset : 0;
    parameter real delay_fb = (t_offset < 0) ? -t_offset : 0;

    // variables
    xbit clk_ref_delay, clk_fb_delay;
    xbit up_xbit, dn_xbit;

    // connectors
    xbit_to_bit #(.width(2)) conn(.in({up_xbit,dn_xbit}), .out({up,dn}));

    // d-flip flop
    delay_xbit #(.delay(delay_ref)) delay_clk_ref(.in(clk_ref), .out(clk_ref_delay));
    delay_xbit #(.delay(delay_fb)) delay_clk_fb(.in(clk_fb), .out(clk_fb_delay));

    dff_xbit #(.delay_cq(clk_to_q_delay)) dff(.d(clk_ref_delay), .clk(clk_fb_delay), .q(up_xbit));

    inv_xbit inv(.in(up_xbit), .out(dn_xbit));

endmodule
/*----------------------------------------------------------------------
MODULE digital_lf.v

= Purpose =

A digital loop filter for PLL.

= Description =

The digital loop filter has a basic PI-controller with an integral gain Ki
and a proportional gain Kp.

    out[n] = Kp*in[n] + Ki*sum(j=1..n-1) in[j]

= Parameters =

||'''NAME'''||'''TYPE'''||'''DESCRIPTION'''||'''UNIT'''||'''DEFAULT'''||
|| init_value || integer || initial control value || None || 10'b10_0000_0000 ||

= Revisions =

$Author$
$DateTIme$
$Id$
----------------------------------------------------------------------*/

module digital_lf (
    input [5:0] up,
    input [5:0] dn,
    input clk,                      // triggering clock
    input reset_n,                  // reset signal
    output reg [9:0] out,           // control output
    output reg signed [2:0] dsm_out,// dsm control output
    output reg clk_lf_dco           // loop-filter clock for the DCO retiming
);

    parameter integer init_value = 10'b10_0000_0000;    // initial control value

    // variables
    wire    [9:0]  out_nxt;
    wire    [9:0]  in_ext;
    wire    [6:0]  in;

    // digital filter
    assign in = {1'b0,up} - {1'b0,dn};
    assign in_ext = {{3{in[6]}},in};

    always @(posedge clk or negedge reset_n) begin
        if (!reset_n) begin
            out <= init_value;
            dsm_out <= 3'b0;
        end
        else begin
            out <= out_nxt;
            dsm_out <= 3'b0;
        end
    end

    assign out_nxt = out + in_ext;

    always @(clk) begin
        clk_lf_dco = clk;
    end

endmodule
/*----------------------------------------------------------------------
MODEL dpll.sv

= Purpose =

dpll model

= Description =

GRO_TDC + D-LF + LCDCO + PRESCALER + DIV8_MPHASE + PH_ROTATOR

= Revisions =

$Authors$
$DateTime$
$Id$

----------------------------------------------------------------------*/

module dpll (
    `input_xbit clk_ref,    // Reference clock from pulse generator
    input reset_n,          // Global reset (active low)
    input vdda,      // Analog power
    input vddt,      // Digital power
    input vss,       // Ground

    input [1:0] tdc_f_range,        // GRO_TDC operating frequency range
    input [1:0] ph_rot_f_range,     // PH_ROTATOR operating frequency range
    input [2:0] pgain_ctrl,         // proportional gain cotrol (DCO)
    input [1:0] div_sel_prescaler,  // dividing factor selection code in PRESCALER
    input [6:0] phsel,              // phase selection code in PH_ROTATOR

    `output_xbit clk_fb,            // Feedback clock(to pad) of PLL
    `output_xbit clk_outp,          // Output clock(to pad) of PLL (positive)
    `output_xbit clk_outn           // Output clock(to pad) of PLL (negative)
);

    // variables
    wire [5:0] up, dn;
    wire clk_lf;
    reg clk_lf_dco;
    reg [9:0] dctrl;
    reg [2:0] dctrl_dsm;

    `xbit(clk_fb_n);
    `xbit(prop_up);
    `xbit(prop_dn);
    `xbit(clk_div_pre_p);
    `xbit(clk_div_pre_n);
    `xbit([7:0] clk_div8_mphase);

    // 1. GRO_TDC --------------------------------------------------------
    gro_tdc gro_tdc (
        .vdd(vdda), .vss(vss),
        .f_range(tdc_f_range),
        .up(up), .dn(dn), .prop_up(prop_up), .prop_dn(prop_dn),
        .clk_ref(clk_ref), .clk_fb(clk_fb), .reset_n(reset_n)
    );

    // 2. digital filter (D-LF) ------------------------------------------
    digital_lf digital_lf (
        .up(up), .dn(dn), .reset_n(reset_n), .clk(clk_lf),
        .out(dctrl), .dsm_out(dctrl_dsm), .clk_lf_dco(clk_lf_dco)
    );

    `ifndef SYN
        xbit_to_bit conn(.in(clk_fb), .out(clk_lf));
    `else
        assign clk_lf = clk_fb;
    `endif

    // 3. LCDCO ----------------------------------------------------------
    lcdco lcdco (
        .vdd(vdda), .vss(vss),
        .clk_lf(clk_lf_dco),
        .dco_pgain(pgain_ctrl), .prop_up(prop_up), .prop_dn(prop_dn),
        .in(dctrl), .in_dsm(dctrl_dsm),
        .reset_n(reset_n), .clk_outp(clk_outp), .clk_outn(clk_outn)
    );

    // 4. PRESCALER ------------------------------------------------------
    prescaler prescaler (
        .vdd(vdda), .vss(vss),
        .clk_inp(clk_outp), .clk_inn(clk_outn),
        .clk_outp(clk_div_pre_p), .clk_outn(clk_div_pre_n),
        .reset_n(reset_n), .div_sel(div_sel_prescaler)
    );

    // 5. DIV8_MPHASE ----------------------------------------------------
    div8_mphase div8_mphase (
        .vdd(vdda), .vss(vss),
        .clk_inp(clk_div_pre_p), .clk_inn(clk_div_pre_n),
        .clk_out(clk_div8_mphase),
        .reset_n(reset_n)
    );

    // 6. PH_ROTATOR -----------------------------------------------------
    ph_rotator ph_rotator (
        .vdd(vdda), .vss(vss),
        .reset_n(reset_n),
        .f_range(ph_rot_f_range),
        .clk_in(clk_div8_mphase), .phsel(phsel),
        .clk_out(clk_fb), .clk_outb(clk_fb_n)
    );

endmodule
/*----------------------------------------------------------------------
MODEL top.sv

= Purpose =

dpll model

= Description =

= Revisions =

$Authors$
$DateTime$
$Id$

----------------------------------------------------------------------*/

module top(
    input vdda,
    input vddt,
    input vss,

    `input_xbit clk_ref_p,
    `input_xbit clk_ref_n,

    input reset_n,

    output clk_fb,
    output clk_ref_out,
    `output_xbit clk_outp,
    `output_xbit clk_outn
);

    wire w_reset_n;  

    `xbit(x_clk_ref);
    wire w_clk_ref;  

    `xbit(x_clk_fb);
    wire w_clk_fb;   

    `xbit(x_clk_outp); 
    `xbit(x_clk_outn); 

    `ifndef SYN
        xbit_to_bit xb2b0( .in(x_clk_ref), .out(w_clk_ref));
        xbit_to_bit xb2b1( .in(x_clk_fb), .out(w_clk_fb));
    `else
        assign w_clk_ref = x_clk_ref;
        assign w_clk_fb = x_clk_fb;
    `endif

    //PAD Define
    //corner cell
    pad_corner cornerul(.vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    pad_corner cornerur(.vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    pad_corner cornerlr(.vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    pad_corner cornerll(.vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));

    //Counter clock wise dierection from left to top point
    // 1. left side ( # inc. down to up) 
    padv_in_cmos     G1_8( .pad(reset_n), .core_in(w_reset_n), .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padv_vss         G1_7(.vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padv_vdda_1      G1_6( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padv_vss         G1_5( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padv_vdda_0      G1_4( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padv_vss         G1_3( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padv_out_cmos_en G1_2( .pad(clk_ref_out), .core_out(w_clk_ref), .out_en(1'b1), .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padv_vss         G1_1( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));

    // 2. top side ( #inc. left to right)
    padh_vss         G2_1( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vddt        G2_2( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vss         G2_3( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vdda_0      G2_4( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vss         G2_5( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vdda_1      G2_6( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vss         G2_7( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vddt        G2_8( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));

    // 3. right side ( #inc. down to up)
    padh_vss         G3_8( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vddt        G3_7( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vss         G3_6( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vdda_0      G3_5( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vss         G3_4( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vddt        G3_3( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vss         G3_2( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vdda_1      G3_1( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));

    // 4. bottom side ( #inc. left to right)
    padh_empty       G4_1( .pad(clk_ref_n), .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_empty       G4_2( .pad(clk_ref_p), .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vss         G4_3( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_out_cmos_en G4_4( .pad(clk_fb), .core_out(w_clk_fb), .out_en(1'b1), .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vss         G4_5( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_empty       G4_6( .pad(clk_outn), .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_empty       G4_7( .pad(clk_outp), .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));
    padh_vss         G4_8( .vdda_0(vdda), .vdda_1(), .vddt(vddt), .vss(vss));

    //custom cell
    in_csda in_csda(
        .vddt(vddt),
        .vss(vss),
        .inp(clk_ref_p),
        .inn(clk_ref_n),
        .out(x_clk_ref)
    );

    out_cml out_cml(
        .vddt(vddt),
        .vss(vss),
        .cmlbias_ctrlb(4'b0000),
        .bias_rstb(w_reset_n),
        .cml_enb(1'b0),
        .inp(x_clk_outp),
        .inn(x_clk_outn),
        .outp(clk_outp),
        .outn(clk_outn)
    );

    dpll dpll0 (
        .clk_ref(x_clk_ref),    
        .reset_n(w_reset_n),        
        .vdda(vdda),       
        .vddt(vddt),     
        .vss(vss),       
        .tdc_f_range(2'b10),
        .ph_rot_f_range(2'b01),
        .pgain_ctrl(3'b111),        
        .div_sel_prescaler(2'b10),  
        .phsel(7'b000_0000),        
        .clk_fb(x_clk_fb),          
        .clk_outp(x_clk_outp),      
        .clk_outn(x_clk_outn)       
    );

endmodule
